import datetime


class Schedule(object):
    _fields = ['mon', 'tue', 'wed', 'thu', 'fri', 'sat', 'sun']

    def __init__(self, make_empty):
        self.data = {}
        for field in self._fields:
            self.data[field] = [0, -1] if make_empty else [0, 23]

    def update(self, data):
        for k, v in data.iteritems():
            self.data[k] = [int(x) for x in v.split('-')]

    def intersect(self, rhs):
        for field in self._fields:
            if rhs.data[field][1] < self.data[field][1]:
                self.data[field][1] = rhs.data[field][1]
            if rhs.data[field][0] > self.data[field][0]:
                self.data[field][0] = rhs.data[field][0]

    def can_run_now(self):
        now = datetime.datetime.now()
        dow = self._fields[now.weekday()]
        hour = now.hour

        limits = self.data[dow]
        return limits[0] <= hour <= limits[1]


class Rules(object):
    def __init__(self, ctx):
        self.ctx = ctx
        self.log = self.ctx.log.getChild('Rules')
        self.cfg = ctx.cfg['rules'] if 'rules' in ctx.cfg else {}
        self.rules = {}
        self.defaultSchedule = None
        self._process_rules()
        self.setDefaultSchedule()

    def _process_rules(self):
        for group, data in self.cfg.iteritems():
            if group == 'default':
                continue
            packs = data['packages']
            sched = Schedule(True)
            sched.update(data['schedule'])
            parallel = int(data['parallel']) if 'parallel' in data else 0

            for pack in packs:
                self.rules[pack] = [sched, parallel]

    def get_parallel(self, packages):
        result = 1000000
        for pack in packages:
            if pack in self.rules:
                pack_parallel = self.rules[pack][1]
                if pack_parallel and pack_parallel < result:
                    result = pack_parallel
        if result < 1000000:
            return result
        else:
            return 0

    def get_schedule(self, packages):
        if not self.rules and not self.defaultSchedule:
            return Schedule(False)

        result = Schedule(False)
        for pack in packages:
            if pack in self.rules:
                result.intersect(self.rules[pack][0])
            elif self.defaultSchedule:
                result.intersect(self.defaultSchedule)
        return result

    def setDefaultSchedule(self):
        if 'default' in self.cfg:
            self.defaultSchedule = Schedule(True)
            self.defaultSchedule.update(self.cfg['default']['schedule'])
