#!/opt/python/bin/python

from __future__ import division, absolute_import

import sys
import os
import ctypes
import errno
import argparse
import logging

from setproctitle import setproctitle
from conductor.agent.cfg import getConfig, getDefaultConfig, serializeConfig
from conductor.agent.log import (setup_logging_to_stdout, setup_logging_to_file, ReopenFileHandler)
from conductor.agent import Supervisor


class Ctx(object):
    def __init__(self):
        # init log to stdout to log errors
        # if we failed to parse config etc.
        setup_logging_to_stdout()
        self.log = logging.getLogger()
        self.logPath = None
        self.cfg = None
        self.version = ''
        self.system = ''

    def initialize(self, argv):
        args = self.parseArgs(argv[1:])

        # Read config
        self.cfg = getConfig(args.cfg, args.cfg_dir)
        if args.print_config:
            print serializeConfig(self.cfg)
            raise SystemExit(0)

        # set process title
        setproctitle(self.cfg['programName'])

        # process command line arguments

        # log configuration part
        self.setupLogging(self.cfg['log'], console=args.console)
        # log what config we're using
        # couldn't do that earlier
        # because log has been just set up
        if args.cfg:
            self.log.info("using '{0}' as config".format(args.cfg))
        else:
            self.log.info("using hardwired config")

        self.readVersion()
        self.readSystemInfo()
        self.addPaths()

        return self

    def parseArgs(self, argv):
        """
        Service command line parsing utility.
        """
        parser = argparse.ArgumentParser(description='Conductor agent service')
        parser.add_argument('-c', '--cfg',
                            default=None,
                            action='store', help='path to service config file')

        parser.add_argument('-d', '--cfg-dir',
                            default=None,
                            action='store', help='path to directory with config files')

        parser.add_argument('--print-config',
                            default=False,
                            action='store_true', help="Parse configs, print result and exit")

        # --console is used to dump logs during debugging with IDE
        # when stdout isn't terminal but we don't won't to log to file
        parser.add_argument('--console',
                            default=False,
                            action='store_true',
                            help='redirect log to stdout (overrides config)')
        return parser.parse_args(argv)

    def setupLogging(self, logCfg, console=False):
        """
        Setup service logs according to config
        """
        logging.raiseExceptions = False
        if not console:
            from conductor.agent.log.handlers import ReopenFileHandler

            self.logPath = logCfg['logPath']
            logFile = os.path.join(self.logPath, logCfg['mainLogFile'])
            handler = ReopenFileHandler(logFile)
            self.log = setup_logging_to_file(handler)

            # don't forget to remove stdout handler
            for h in self.log.handlers:
                self.log.removeHandler(h)

        # python2.6 doesn't automagically converts level name to int
        # so we do it manually
        loglevel = logging.getLevelName(logCfg['logLevel'])
        self.log.setLevel(loglevel)

        logging.getLogger('kazoo').setLevel(logging.INFO)

    def readVersion(self):
        if os.path.exists('/etc/conductor-agent/version'):
            with open('/etc/conductor-agent/version', 'r') as f:
                self.version = f.read().strip()

    def readSystemInfo(self):
        if os.path.exists('/etc/os-release'):
            with open('/etc/os-release') as f:
                self.system = f.readline().strip().split('=', 1)[1].replace('"', '') + ' '
                self.system += f.readline().strip().split('=', 1)[1].replace('"', '')
        if os.path.exists('/etc/lsb-release'):
            with open('/etc/lsb-release') as f:
                for l in f.readlines():
                    if l.startswith("DISTRIB_DESCRIPTION"):
                        self.system = l.strip().split('=', 1)[1].replace('"', '')
        if os.path.exists('/etc/redhat-release'):
            with open('/etc/redhat-release') as f:
                self.system = f.read().strip()

    def addPaths(self):
        for p in self.cfg.get('pythonpath', []):
            sys.path.append(p)


def formatExc(msg, e):
    args = msg, e.__class__.__name__, " ".join(str(i) for i in e.args)
    return "{0}: type='{1}' args='{2}'".format(*args)


def main(argv):
    exitCode = 123  # Something weird happened
    agent = None
    ctx = Ctx()
    try:
        # make a context
        ctx.initialize(argv)

        # some daemon specific actions if needed
        os.chdir('/')

        # Create, run and wait for agent to end it's work
        agent = Supervisor(ctx).start().join()

    # some handy exception handling
    # to show nicely formatted and detailed enough
    # error messages to user
    except SystemExit:
        raise
    except KeyboardInterrupt:
        ctx.log.info("exiting on user request")
        exitCode = 0
    # IOError has 'filename' attribute so we handle it
    # separately from Exception
    except IOError as e:
        errorCode = errno.errorcode.get(e.errno, e.errno)
        ctx.log.exception("exiting on io error: file='{0}' code='{1}' msg='{2}'".format(
            e.filename, errorCode, e.strerror))
        exitCode = 1
    except Exception as e:
        # BaseException.message is deprecated so we manually
        # construct nice looking (hopefully) string to log
        ctx.log.exception(formatExc("exiting on error", e))
        exitCode = 1
    else:
        exitCode = 0
    finally:
        if agent is not None:
            agent.stop().join()
    raise SystemExit(exitCode)


if __name__ == '__main__':
    main(sys.argv)
