#pragma once
#include "../../../config.h"
#include "../../utility.hh"
#include "../../dnsbackend.hh"
#include "../../dns.hh"
#include "../../dnsbackend.hh"
#include "../../dnspacket.hh"
#include "../../pdnsexception.hh"
#include "../../logger.hh"
#include "../../version.hh"
#include <boost/algorithm/string.hpp>
#include <util/generic/string.h>

namespace {
    TString StringToStroka(const std::string& s) {
        return TString(s.c_str(), s.length());
    }

    std::string StrokaToString(const TString& s) {
        return std::string(s.data(), s.size());
    }
} // namespace
/* FIRST PART */
class RandomBackend : public DNSBackend
{
public:
  RandomBackend(const string &suffix="")
  {
    setArgPrefix("random"+suffix);
    d_ourname=DNSName("yandex.ru");
    d_ourdomain = d_ourname;
    d_ourdomain.chopOff();
  }

  bool list(const DNSName &target, int id, bool include_disabled) override {
    return false; // we don't support AXFR
  }

  void lookup(const QType &type, const DNSName &qdomain, DNSPacket *p, int zoneId) override
  {
    if(qdomain == d_ourdomain){
      if(type.getCode() == QType::SOA || type.getCode() == QType::ANY) {
        d_answer="ns1." + StringToStroka(d_ourdomain.toString()) + " hostmaster." + StringToStroka(d_ourdomain.toString()) + " 1234567890 86400 7200 604800 300";
      } else {
        d_answer.clear();;
      }
    } else if (qdomain == d_ourname) {
      if(type.getCode() == QType::A || type.getCode() == QType::ANY) {
        ostringstream os;
        os<<Utility::random()%256<<"."<<Utility::random()%256<<"."<<Utility::random()%256<<"."<<Utility::random()%256;
        d_answer=StringToStroka(os.str()); // our random ip address
      } else {
        d_answer="";
      }
    } else {
      d_answer="";
    }
  }

  bool get(DNSResourceRecord &rr) override
  {
    if(d_answer.empty())
      return false;

    if(d_answer.find("ns1.") == 0){
      rr.qname=d_ourdomain;
      rr.qtype=QType::SOA;
    } else {
      rr.qname=d_ourname;
      rr.qtype=QType::A;
    }
    rr.qclass=QClass::IN;   // Internet class randomness.
    rr.ttl=5;               // 5 seconds
    rr.auth = 1;            // it may be random.. but it is auth!
    rr.content = StrokaToString(d_answer);

    d_answer.clear();       // this was the last answer
    return true;
  }

private:
  TString d_answer;
  DNSName d_ourname;
  DNSName d_ourdomain;
};

/* SECOND PART */

class RandomFactory : public BackendFactory
{
public:
  RandomFactory() : BackendFactory("random") {}
  void declareArguments(const string &suffix="")
  {
    declare(suffix,"hostname","Hostname which is to be random","random.example.com");
  }
  DNSBackend *make(const string &suffix="")
  {
    return new RandomBackend(suffix);
  }
};

/* THIRD PART */

class RandomLoader
{
public:
  RandomLoader()
  {
    BackendMakers().report(new RandomFactory);
    PDNS_LOG<< Logger::Info << "[randombackend] This is the random backend version " VERSION
#ifndef REPRODUCIBLE
      << " (" __DATE__ " " __TIME__ ")"
#endif
      << " reporting" << endl;
  }  
};

static RandomLoader randomLoader;
