from __future__ import unicode_literals
import os
import logging

import flask
import raven.conf
import raven.handlers.logging
from gevent import pywsgi
import gevent

import yp.client
from sepelib.core import config
from infra.rsc.src.model import yp_client
from infra.swatlib import metrics
from infra.controllers_monitoring.src import yp_poller
from infra.controllers_monitoring.src.lib import sockutil

log = logging.getLogger('main')


def failure(message, *args, **kwargs):
    log.error(message, *args, **kwargs)
    raise SystemExit(1)


def make_client_map(config):
    rv = {}
    for c in config.get_value('yp.clusters'):
        cluster = c['cluster']
        address = c['address']
        token = c.get('token') or os.environ.get('YP_TOKEN')
        base = yp.client.YpClient(address=address, config={'token': token})
        stub = base.create_grpc_object_stub()
        rv[cluster] = yp_client.YpClient(stub=stub)
    return rv


class Application(object):

    name = 'controllers-monitoring'

    def __init__(self, instance_id):
        sentry_dsn = config.get_value('sentry.dsn', default=None)
        # Web application initialization
        if sentry_dsn:
            self.setup_logging(sentry_dsn=sentry_dsn)

    @staticmethod
    def setup_logging(sentry_dsn):
        c = raven.Client(dsn=sentry_dsn)
        c.transport_options['maximum_outstanding_requests'] = 50
        handler = raven.handlers.logging.SentryHandler(c, level=logging.WARN)
        raven.conf.setup_logging(handler)

    @staticmethod
    def setup_environment():
        # Patch requests connection pool to use gevent queue
        from requests.packages.urllib3.connectionpool import ConnectionPool
        from gevent.queue import LifoQueue

        ConnectionPool.QueueCls = LifoQueue
        # Disable requests spamming about:
        # Connection pool is full, discarding connection
        # This is the way we use alemate-http to avoid broken connections
        # There is nothing we can do about it, so simply mute
        logging.getLogger('requests.packages.urllib3.connectionpool').setLevel(logging.ERROR)

    def teardown_environment(self):
        pass

    def init_http_server(self, registry):
        app = flask.Flask('controllers_monitoring')

        metrics.MetricsExt(app, registry=registry, cfg={
            'yasm_stat_url': '/stat',
            'performance_enabled': False,
            'status_code_enabled': False,
            'export_current_timestamp': False
        })

        c = config.get_value('web').get('http')
        host = c.get('host', '::')
        port = c['port']

        log.info('Binding [%s]:%d for HTTP server...', host, port)
        sock, err = sockutil.create_server_socket(host, port)
        if err is not None:
            failure('Failed to init server socket: %s', err)
        # Use gevent http server
        return pywsgi.WSGIServer(sock, app)

    def run_server(self, r):
        log.info('Starting HTTP server...')
        self.server = self.init_http_server(registry=r)
        self.server.serve_forever()

    def run_poller(self, r, batch_size, pod_match_dict, rs_match_dict, yasm_export_mcrs_ids):
        gevent.sleep(0.5)
        log.info('Starting poller...')
        self.poller = yp_poller.YpPoller(
            metrics_registry=r,
            client_map=make_client_map(config),
            xdc_cluster=config.get_value('yp.xdc'),
            batch_size=batch_size,
            pod_match_dict=pod_match_dict,
            rs_match_dict=rs_match_dict,
            yasm_export_mcrs_ids=yasm_export_mcrs_ids,
        )
        self.poller.run()

    def run(self):
        """
        Start application.
        """
        log.info('starting service...')
        self.setup_environment()

        r = metrics.Registry()

        batch_size = config.get_value('yp.poll.batch_size')
        pod_match_dict = config.get_value('pod.match_labels')
        rs_match_dict = config.get_value('rs.match_labels')
        yasm_export_mcrs_ids = config.get_value('mcrs.db_exceeded_export.deploy_units', default=[])
        threadpool = gevent.threadpool.ThreadPool(1)
        self.server_greenlet = threadpool.apply_async(self.run_server, (r, ))
        try:
            self.run_poller(r, batch_size, pod_match_dict, rs_match_dict, yasm_export_mcrs_ids)
            self.server_greenlet.get()
        except KeyboardInterrupt:
            self.stop()

    def stop(self):
        """
        Gracefully stop application.
        """
        log.info('Stopping HTTP server...')
        gevent.kill(self.server_greenlet)
        log.info('Stopping metrics collector..')
        self.poller.stop()
        log.info('Done.')
