from __future__ import unicode_literals
import collections

import yp.data_model

from infra.swatlib.gevent import geventutil


class Mcrs(object):
    def __init__(self):
        self.mcrs = None
        self.pods = collections.defaultdict(list)

    def is_empty(self, unavailable_clusters):
        for cluster in self.mcrs.spec.clusters:
            if cluster.cluster in unavailable_clusters:
                continue
            if cluster.spec.replica_count != 0:
                return False
        return True

    def is_ready(self):
        return self.mcrs.status.ready.status == yp.data_model.CS_TRUE

    def is_in_progress(self):
        return self.mcrs.status.in_progress.status == yp.data_model.CS_TRUE

    def is_failed(self):
        return self.mcrs.status.failed.status == yp.data_model.CS_TRUE

    def iter_pods(self):
        for cluster, pods in self.pods.iteritems():
            for p in pods:
                yield p

    def is_status_up_to_date(self, unavailable_clusters):
        if self.is_empty(unavailable_clusters):
            return True
        return self.mcrs.spec.revision == self.mcrs.status.revision

    def has_up_to_date_pods(self, unavailable_clusters):
        if self.is_empty(unavailable_clusters):
            return True
        r = self.mcrs.spec.revision
        for p in geventutil.gevent_idle_iter(self.iter_pods()):
            if p.spec.pod_agent_payload.spec.revision == r:
                return True
        return False

    def is_replica_count_matched(self, unavailable_clusters):
        for c in self.mcrs.spec.clusters:
            if c.cluster in unavailable_clusters:
                continue
            if c.spec.replica_count != len(self.pods.get(c.cluster, [])):
                return False
        return True

    def is_controller_status_success(self):
        s = self.mcrs.status.multi_cluster_deploy_status.details.controller_status
        return s.last_attempt.succeeded.status == yp.data_model.CS_TRUE

    def unready_pods_over_budget_count(self):
        progress = self.mcrs.status.multi_cluster_deploy_status.details.total_progress
        unavailable = progress.pods_total - progress.pods_ready
        return max(0, unavailable - self.mcrs.spec.deployment_strategy.max_unavailable)


def collect_metrics(xdc_cluster, registry, mcrs_list, unavailable_clusters, clusters_list, yasm_export_mcrs_ids=None):
    total_mcrs_count = 0
    ready_rs_status = 0
    in_progress_rs_status = 0
    failed_rs_status = 0
    status_up_to_date = 0
    old_status = 0
    pod_up_to_date = 0
    old_pod = 0
    matched_replica_count = 0
    unmatched_replica_count = 0
    controller_status_success = 0
    controller_status_fail = 0
    budget_exceeded_mcrs_count = 0

    for mcrs in geventutil.gevent_idle_iter(mcrs_list):
        if not mcrs.mcrs:
            continue

        total_mcrs_count += 1
        if mcrs.is_status_up_to_date(unavailable_clusters):
            status_up_to_date += 1
        else:
            old_status += 1
        if not yasm_export_mcrs_ids or mcrs.mcrs.meta.id in yasm_export_mcrs_ids:
            unready_pods_over_budget = mcrs.unready_pods_over_budget_count()
            tags = {
                "deploy_unit": mcrs.mcrs.meta.id
            }
            registry.get_gauge('unready_pods_over_budget', 'txxx', tags=tags).set(unready_pods_over_budget)
            if unready_pods_over_budget > 0:
                budget_exceeded_mcrs_count += 1
        if mcrs.is_failed():
            failed_rs_status += 1
        elif mcrs.is_in_progress():
            in_progress_rs_status += 1
        elif mcrs.is_ready():
            ready_rs_status += 1

        if mcrs.has_up_to_date_pods(unavailable_clusters):
            pod_up_to_date += 1
        else:
            old_pod += 1
        if mcrs.is_replica_count_matched(unavailable_clusters):
            matched_replica_count += 1
        else:
            unmatched_replica_count += 1
        if mcrs.is_controller_status_success():
            controller_status_success += 1
        else:
            controller_status_fail += 1

    for cluster in clusters_list:
        is_available = int(cluster not in unavailable_clusters)
        registry.get_gauge('is_available_{}_mcrsc'.format(cluster), 'axxx').set(is_available)
    registry.get_gauge('total_count_{}_mcrsc'.format(xdc_cluster), 'axxx').set(total_mcrs_count)
    registry.get_gauge('total_budget_exceeded_{}_mcrsc'.format(xdc_cluster), 'axxx').set(budget_exceeded_mcrs_count)
    registry.get_gauge('in_progress_status_{}_mcrsc'.format(xdc_cluster), 'axxx').set(in_progress_rs_status)
    registry.get_gauge('failed_status_{}_mcrsc'.format(xdc_cluster), 'axxx').set(failed_rs_status)
    registry.get_gauge('status_up_to_date_{}_mcrsc'.format(xdc_cluster), 'annn').set(status_up_to_date)
    registry.get_gauge('old_status_{}_mcrsc'.format(xdc_cluster), 'axxx').set(old_status)
    registry.get_gauge('pod_up_to_date_{}_mcrsc'.format(xdc_cluster), 'annn').set(pod_up_to_date)
    registry.get_gauge('old_pod_{}_mcrsc'.format(xdc_cluster), 'axxx').set(old_pod)
    registry.get_gauge('matched_replica_count_{}_mcrsc'.format(xdc_cluster), 'annn').set(matched_replica_count)
    registry.get_gauge('unmatched_replica_count_{}_mcrsc'.format(xdc_cluster), 'axxx').set(unmatched_replica_count)
    registry.get_gauge('controller_status_success_{}_mcrsc'.format(xdc_cluster), 'annn').set(controller_status_success)
    registry.get_gauge('controller_status_fail_{}_mcrsc'.format(xdc_cluster), 'axxx').set(controller_status_fail)
