from __future__ import unicode_literals
import collections

import yp.data_model

from infra.swatlib.gevent import geventutil


class Rs(object):
    def __init__(self):
        self.rs = None
        self.pods = []

    def is_empty(self):
        return self.rs.spec.replica_count == 0

    def is_ready(self):
        return self.rs.status.ready_condition.status == yp.data_model.CS_TRUE

    def is_in_progress(self):
        return self.rs.status.in_progress_condition.status == yp.data_model.CS_TRUE

    def is_failed(self):
        return self.rs.status.failed_condition.status == yp.data_model.CS_TRUE

    def is_status_up_to_date(self):
        if self.is_empty():
            return True
        return self.rs.spec.revision_id == self.rs.status.revision_id

    def has_up_to_date_pods(self):
        if self.is_empty():
            return True

        r = int(self.rs.spec.revision_id)
        for p in geventutil.gevent_idle_iter(self.pods):
            if p.spec.pod_agent_payload.spec.revision == r:
                return True
        return False

    def is_replica_count_matched(self):
        return self.rs.spec.replica_count == len(self.pods)

    def is_controller_status_success(self):
        s = self.rs.status.deploy_status.details.controller_status
        return s.last_attempt.succeeded.status == yp.data_model.CS_TRUE

    def unready_pods_over_budget_count(self):
        progress = self.rs.status.deploy_status.details.total_progress
        unavailable = progress.pods_total - progress.pods_ready
        return max(0, unavailable - self.rs.spec.deployment_strategy.max_unavailable)


def collect_metrics(cluster, registry, rs_list, yasm_export_mcrs_ids):
    total_rs_count = 0
    ready_rs_status = 0
    in_progress_rs_status = 0
    failed_rs_status = 0
    status_up_to_date = 0
    old_status = 0
    pod_up_to_date = 0
    old_pod = 0
    matched_replica_count = 0
    unmatched_replica_count = 0
    controller_status_success = 0
    controller_status_fail = 0
    budget_exceeded_rs_count = 0
    unready_pods_over_budget_rs_dict = {}

    for rs in geventutil.gevent_idle_iter(rs_list):
        if not rs.rs:
            continue

        total_rs_count += 1
        if rs.is_status_up_to_date():
            status_up_to_date += 1
        else:
            old_status += 1
        if not yasm_export_mcrs_ids or rs.rs.meta.id in yasm_export_mcrs_ids:
            unready_pods_over_budget = rs.unready_pods_over_budget_count()
            tags = {
                "deploy_unit": rs.rs.meta.id,
            }
            gauge = registry.get_gauge('unready_pods_over_budget_{}'.format(cluster), 'txxx', tags=tags)
            gauge.set(unready_pods_over_budget)
            unready_pods_over_budget_rs_dict[rs.rs.meta.id] = unready_pods_over_budget
            if unready_pods_over_budget > 0:
                budget_exceeded_rs_count += 1

        if rs.is_failed():
            failed_rs_status += 1
        elif rs.is_in_progress():
            in_progress_rs_status += 1
        elif rs.is_ready():
            ready_rs_status += 1

        if rs.has_up_to_date_pods():
            pod_up_to_date += 1
        else:
            old_pod += 1
        if rs.is_replica_count_matched():
            matched_replica_count += 1
        else:
            unmatched_replica_count += 1
        if rs.is_controller_status_success():
            controller_status_success += 1
        else:
            controller_status_fail += 1

    registry.get_gauge('total_budget_exceeded_{}_rsc'.format(cluster), 'axxx').set(budget_exceeded_rs_count)
    registry.get_gauge('total_count_{}_rsc'.format(cluster), 'axxx').set(total_rs_count)
    registry.get_gauge('ready_status_{}_rsc'.format(cluster), 'axxx').set(ready_rs_status)
    registry.get_gauge('in_progress_status_{}_rsc'.format(cluster), 'axxx').set(in_progress_rs_status)
    registry.get_gauge('failed_status_{}_rsc'.format(cluster), 'axxx').set(failed_rs_status)
    registry.get_gauge('status_up_to_date_{}_rsc'.format(cluster), 'annn').set(status_up_to_date)
    registry.get_gauge('old_status_{}_rsc'.format(cluster), 'axxx').set(old_status)
    registry.get_gauge('pod_up_to_date_{}_rsc'.format(cluster), 'annn').set(pod_up_to_date)
    registry.get_gauge('old_pod_{}_rsc'.format(cluster), 'axxx').set(old_pod)
    registry.get_gauge('matched_replica_count_{}_rsc'.format(cluster), 'annn').set(matched_replica_count)
    registry.get_gauge('unmatched_replica_count_{}_rsc'.format(cluster), 'axxx').set(unmatched_replica_count)
    registry.get_gauge('controller_status_success_{}_rsc'.format(cluster), 'annn').set(controller_status_success)
    registry.get_gauge('controller_status_fail_{}_rsc'.format(cluster), 'axxx').set(controller_status_fail)
    return unready_pods_over_budget_rs_dict


def set_deploy_units_budget_metrics(cluster_results, registry):
    unready_pods_over_budget_rs_dict = collections.defaultdict(int)
    for cluster, result in cluster_results.iteritems():
        for deploy_unit, value in result.unready_pods_over_budget_rs_dict.iteritems():
            unready_pods_over_budget_rs_dict[deploy_unit] += value
    for deploy_unit, unready_pods_over_budget in unready_pods_over_budget_rs_dict.iteritems():
        tags = {
            "deploy_unit": deploy_unit
        }
        gauge = registry.get_gauge('unready_pods_over_budget', 'txxx', tags=tags)
        gauge.set(unready_pods_over_budget)
