import copy
import datetime
import json
import os
import re
import time
import logging

from dateutil import parser

from sqlalchemy import and_, or_, tuple_

import sandbox.projects.release_machine.core.const as rm_const
from sandbox.projects.common.nanny.client import NannyClient

from library.python import coredump_filter

from infra.cores.app import const
from infra.cores.app import db
from infra.cores.app import date_time
from infra.cores.app import filters
from infra.cores.app import garbage_collector
from infra.cores.app import models
from infra.cores.app import notifications
from infra.cores.app import tables
from infra.cores.app import strings
from infra.cores.app import suggest

from infra.cores.app.models import (
    Core,
    CoreSingle,
    CoreDetails,
    UserInfo,
)


MODEL = {
    "first_time": Core.first_time,
    "last_time": Core.last_time,
    "core_hash": Core.core_hash,
    "core_summary": Core.core_summary,
    "itype": Core.itype,
    "full_trace": CoreSingle.full_trace,
    "signal": Core.signal,
    "count": Core.count,
    "ctype_list": Core.ctype_list,
    "core_id": CoreDetails.core_id,
    "core_d_hash": CoreDetails.core_hash,
    "top_frame": CoreDetails.top_frame,
    "instance": CoreDetails.instance,
    "ctype": CoreDetails.ctype,
    "sb_build": CoreDetails.sb_build,
    "sb_task_run": CoreDetails.sb_task_run,
    "timestamp": CoreDetails.timestamp,
    "core_single_id": CoreSingle.core_id,
    "user_name": UserInfo.user_name,
    "prj": CoreDetails.prj,
    "prj_list": Core.prj_list,
    "host_list": Core.host_list,
}


def get_core_summary(core_parsed):
    return core_parsed[0][0].simple_html(5)


def get_core_summary_raw(raw_core):
    return raw_core[0][0]


get_top_frame_raw = get_core_summary_raw


def get_top_frame(core_parsed):
    return core_parsed[0][0].simple_html()


def get_tags(tags_str):
    return [tag.strip() for tag in re.split(r",|\|", tags_str) if tag.strip()]


def get_nanny_service_admins(nanny_service_id):
    nn_client = NannyClient(api_url=rm_const.Urls.NANNY_BASE_URL, oauth_token=os.getenv("TESTENV_API_OAUTH"))
    auth_response = nn_client.get_service_auth_attrs(service_id=nanny_service_id)
    logging.info("AUTH_RESPONSE: %s", auth_response)


def get_items_for_main_table_suggest(query):
    suggest_items = suggest.get_unique_items(
        query,
        [
            "itype",
            "ctype_list",
            "prj_list",
            "host_list",
            "tag_list",
            "signal",
        ],
        models.Core,
        limit=2000,
    )
    suggest_items["signal"] = [
        "{name} ({num})".format(
            name=const.SIG_NUM_TO_SIG_NAME[signal],
            num=signal,
        ) if signal in const.SIG_NUM_TO_SIG_NAME else signal
        for signal in suggest_items["signal"]
    ]

    for column in suggest_items:
        items = suggest_items[column]
        logging.info(
            "Items for suggest for `%s` column: %s ..., total %s",
            column, list(items[:10]), len(items),
        )

    return suggest_items


def add_core(core_text_full, args, additional_text):
    """
    Process core, send notifications about core in relevant chats.
    :param core_text_full: str, python traceback or gdb-parsed coredump
    :param args: dict, params for core such as itype, ctype, signal and other
    :param additional_text: str, some additional info
    :return: dict with `core_id`, `core_hash`, `core_count`, `is_new_core` fields
    """
    prepared_args = args
    _prepare_args(prepared_args)
    logging.debug("PREPARED ARGS:\n%s", json.dumps(prepared_args, indent=4))
    parse_error = prepared_args.get("parse_error", "")
    parse_result = prepared_args.get("parse_result", "")
    signal = "UNKNOWN"
    got_error = False
    core_text_full = core_text_full.strip()
    if parse_result == "" or parse_result == "ParseResult.OK":
        try:
            if core_text_full.startswith("Traceback (most recent call last):"):
                core_parsed, raw_core, signal = coredump_filter.parse_python_traceback(core_text_full)
            else:
                core_parsed, raw_core, signal = coredump_filter.filter_stack_dump(core_text=core_text_full)
            core_summary = get_core_summary(core_parsed)
            core_summary_raw = get_core_summary_raw(raw_core)
            top_frame = get_top_frame(core_parsed)
            top_frame_raw = get_top_frame_raw(raw_core)
            core_hash = core_parsed[0][0].hash(8)
            logging.info("Core parsing OK, hash: %s", core_hash)
        except Exception as exc:
            logging.exception("Got exception while parsing core: %s", str(exc))
            core_parsed = "Unparsed core"
            raw_core = "Unparsed core"
            core_summary = "Can't parse core"
            core_summary_raw = "Can't parse core"
            top_frame = "Can't parse core"
            top_frame_raw = "Can't parse core"
            core_hash = hash(core_parsed)
            got_error = True
    else:
        logging.error("Core parsing error: %s", parse_error)
        core_parsed = parse_error
        raw_core = parse_error
        core_summary = parse_error
        top_frame = parse_error
        core_summary_raw = parse_error
        top_frame_raw = parse_error
        core_hash = hash(core_parsed)
        got_error = True

    signal = prepared_args.get("signal") or signal

    # logging.debug("Got core_parsed: %s\nraw_core: %s", core_parsed, raw_core)

    ctype = prepared_args.get("ctype", "")
    prj = prepared_args.get("prj", "")
    tags = prepared_args.get("tags", "")
    old_cores = prepared_args.get("old_cores", False)
    if tags:
        tags_prepared = get_tags(tags)
        tags = ",".join(tags_prepared)
    timestamp = _normalize_time(prepared_args.get("time"))
    itype = prepared_args.get("itype", "")
    instance = prepared_args.get("instance", "")
    is_new_core = False

    core = get_core(core_hash, itype)

    if signal:
        signal = str(signal).strip()
    signal = signal if signal in const.SIG_NUM_TO_SIG_NAME.keys() else const.SIG_NAME_TO_SIG_NUM.get(signal, "")

    if not core:
        logging.info("Adding new core for (%s, %s)", core_hash, itype)
        db.session.add(models.Core(
            core_hash=core_hash,
            core_summary=core_summary,
            core_summary_raw=core_summary_raw,
            itype=itype,
            ctype_list=ctype,
            prj_list=prj,
            host_list=instance,
            tag_list=tags,
            signal=signal,
            count=1,
            first_time=timestamp,
            last_time=timestamp,
            tickets="",
            fixed=False,
        ))
    else:
        logging.info("Updating existing core for (%s, %s)", core_hash, itype)
        core.ctype_list = strings.merge_tokens(core.ctype_list, ctype)
        core.prj_list = strings.merge_tokens(core.prj_list, prj)
        core.host_list = strings.merge_tokens(core.host_list, instance)
        core.tag_list = strings.merge_tokens(core.tag_list, tags)
        core.last_time = timestamp
        core.count += 1
        if not core.core_summary_raw:
            core.core_summary_raw = core_summary_raw
        if core.fixed is True:
            is_new_core = True
        core.fixed = False

    db.session.commit()

    get_core_details = models.CoreDetails.query.filter(
        models.CoreDetails.core_hash == core_hash,
        models.CoreDetails.itype == itype,
    )
    total_alive_cores = get_core_details.count()
    if total_alive_cores == 0:
        is_new_core = True

    core_to_add = models.CoreDetails(
        core_hash=core_hash,
        top_frame=top_frame,
        top_frame_raw=top_frame_raw,
        instance=instance,
        itype=itype,
        ctype=ctype,
        prj=prj,
        tags=tags,
        signal=signal,
        parse_error=parse_error,
        sb_build=prepared_args.get("sb_build", None),
        sb_task_run=prepared_args.get("sb_task_run", None),
        timestamp=timestamp,
        tickets="",
        old_cores=old_cores,
        expire_time=garbage_collector.calculate_expire_time(core_hash, itype, timestamp),
    )
    db.session.add(core_to_add)

    core = get_core(core_hash, itype)
    core.last_core_id = core_to_add.core_id
    db.session.commit()

    db.session.add(models.CoreSingle(
        core_id=core_to_add.core_id,
        full_trace=coredump_filter.serialize_stacks(core_parsed),
        full_trace_raw="\n".join(
            [
                "\n".join([stack for stack in stacks_common_hash])
                for stacks_common_hash in raw_core
            ]
        ),
        additional_text=additional_text,
        error=got_error,
    ))
    db.session.commit()

    update_core_tags_table(tags, str(core_to_add.core_id))

    staff_logins = prepared_args.get("staff_logins", "")
    notifications.send_notifications(core_to_add.core_id, itype, instance, staff_logins, core_hash)

    return {
        'core_id': core_to_add.core_id,
        'core_hash': core_hash,
        'core_count': core.count,
        'is_new_core': is_new_core,
    }


def _normalize_time(timestamp):
    """Users can submit time in strange formats."""
    if not timestamp:
        return date_time.icurrent()
    return int(timestamp)


def update_core_tags_table(tags, core_id):
    if not tags:
        return
    tag_list = strings.parse_tokens(tags)
    for tag in tag_list:
        tag_name = models.TagNames.query.filter(
            models.TagNames.tag_name == tag,
        ).first()
        if tag_name:
            tag_id = tag_name.tag_id
        else:
            new_tag = models.TagNames(tag_name=tag)
            db.session.add(new_tag)
            db.session.commit()
            tag_id = new_tag.tag_id
        db.session.add(models.CoreTag(
            tag_id=tag_id,
            core_id=core_id,
        ))
        db.session.commit()


def from_param_to_db(param):
    if param in ["first_time", "last_time"]:
        return models.CoreDetails.timestamp
    if param in ["ctype"]:
        return models.CoreDetails.ctype
    if param in ["itype"]:
        return models.Core.itype
    if param in ["signal"]:
        return models.Core.signal


def get_user_name(request):
    login = request.cookies.get("yandex_login")
    logging.debug("Got login from cookies %s", login)
    if not login:
        login = const.DEVELOPER_LOGIN
    logging.debug("Got user_name %s", login)
    return login


def get_user_cores_types(user_name):
    types = models.UserInfo.query.with_entities(
        models.UserInfo.itype,
        models.UserInfo.ctype,
        models.UserInfo.prj,
        models.UserInfo.tag,
    ).filter_by(user_name=user_name).all()
    types = [(core_type[0], core_type[1], core_type[2], core_type[3]) for core_type in types]  # TODO: remove indices
    logging.info("Found itypes {types} for user {user_name}".format(types=types, user_name=user_name))
    return types


def add_core_type(user_name, core_itype, core_ctype, core_prj, core_tag):
    query = models.UserInfo.query.filter(
        models.UserInfo.user_name == user_name,
        models.UserInfo.itype == core_itype,
        models.UserInfo.ctype == core_ctype,
        models.UserInfo.prj == core_prj,
        models.UserInfo.tag == core_tag,
    ).first()
    if not query:
        db.session.add(models.UserInfo(
            user_name=str(user_name),
            itype=str(core_itype),
            ctype=str(core_ctype),
            prj=str(core_prj),
            tag=str(core_tag),
        ))
        db.session.commit()
        return True
    else:
        logging.debug("There is the same core_type")
        return False


def remove_core_type(user_name, core_itype, core_ctype, core_prj, core_tag):
    logging.debug(
        "Delete core_type {core_type} from user {user_name} settings".format(core_type=core_itype, user_name=user_name)
    )
    db.session.query(models.UserInfo).filter(
        models.UserInfo.user_name == user_name,
        models.UserInfo.itype == core_itype,
        models.UserInfo.ctype == core_ctype,
        models.UserInfo.prj == core_prj,
        models.UserInfo.tag == core_tag,
    ).delete()
    db.session.commit()


def put_cores_in_table(query, page, table_type, page_name=None):
    tmp_time = time.time()
    pager = query.paginate(page=int(page), per_page=const.LIMIT_PER_PAGE)
    logging.debug("Paginate query for main: {}".format(time.time() - tmp_time))

    tmp_time = time.time()
    table_with_cores = [table_type(core_db, page_name) if page_name else table_type(core_db) for core_db in pager.items]
    logging.debug("First query for main: {}".format(time.time() - tmp_time))

    tmp_time = time.time()
    cores_num = pager.total
    logging.debug("Second query for main: {}".format(time.time() - tmp_time))
    if int(cores_num) == 0:
        return None, None
    return table_with_cores, cores_num


def _build_filter_query(args):
    warnings = []
    query = models.Core.query
    request_filters = args[const.FILTER]
    logging.debug("FILTERS: %s", request_filters)
    for filter_key, filter_value in request_filters.items():
        if filter_key == "show_fixed":
            if not filter_value:
                query = query.filter(models.Core.fixed == False)  # noqa

        if not filter_value:
            continue

        if filter_key == "first_time":
            filter_value_converted = time.mktime(parser.parse(filter_value).timetuple())
            query = query.filter(models.Core.first_time > filter_value_converted)
        if filter_key == "last_time":
            filter_value_converted = time.mktime(parser.parse(filter_value).timetuple())
            query = query.filter(models.Core.last_time < filter_value_converted)

        if filter_key == "signal":
            if filter_value.split('(')[0].strip() in const.SIG_NAME_TO_SIG_NUM:
                filter_value = const.SIG_NAME_TO_SIG_NUM[filter_value.split('(')[0].strip()]
            query = query.filter(models.Core.signal == filter_value)

        if filter_key == "itype":
            query = query.filter(models.Core.itype == filter_value)

        if filter_key == "ctype_list":
            query = query.filter(models.Core.ctype_list.contains(filter_value))

        if filter_key == "prj_list":
            query = query.filter(models.Core.prj_list.contains(filter_value))
        if filter_key == "host_list":
            query = query.filter(models.Core.host_list.contains(filter_value))
        if filter_key == "text_to_search":
            if (
                request_filters.get("itype")
            ):
                query = query.filter(models.Core.core_summary_raw.like("%{}%".format(filter_value)))
            else:
                warnings.append(
                    "Пожалуйста, укажите `itype`, чтобы воспользоваться полнотекстовой фильтрацией. "
                )
                logging.warning("Text filters without itype selectors called")

        if filter_key == "tag_list":
            tags = get_tags(filter_value)
            filter_core_ids = filters.get_core_ids_for_tags_filter(
                tags, request_tags_argument=request_filters["tag_list"])
            core_details_query = models.CoreDetails.query.with_entities(
                models.CoreDetails.itype,
                models.CoreDetails.core_hash,
            ).filter(models.CoreDetails.core_id.in_(filter_core_ids)).distinct().all()
            logging.debug("Core hashes %s", [c[0] for c in core_details_query])
            query = query.filter(tuple_(models.Core.itype, models.Core.core_hash).in_(core_details_query))

    return query, warnings


def _add_filter_sort_direction(args, query):
    if args[const.MAIN]["direction"] == "asc":
        query = query.order_by(MODEL[args[const.MAIN]["sort"]])
    else:
        query = query.order_by(MODEL[args[const.MAIN]["sort"]].desc())
    return query


def get_aggr_user_cores(args, types):
    """
    Get all cores for user_cores page
    :param args: dict, with additional filters.
    :param types: list of 4-tuples - (itype, ctype, prj, tag).
    Some of them can be empty strings. We get them from user settings.
    :return: table with cores and items to suggest (prj, tag, etc.)
    """
    if not types:
        return None, None, []

    logging.debug("Start `get_aggr_user_cores` query, reqid={reqid}".format(reqid=args["reqid"]))
    query, warnings = _build_filter_query(args)

    conditions = (
        and_(
            models.Core.itype == itype if itype else True,
            models.Core.ctype_list.contains(ctype),
            models.Core.prj_list.contains(prj),
            models.Core.tag_list.contains(tag),
        )
        for itype, ctype, prj, tag in types
    )
    query = query.filter(or_(*conditions))

    suggest_items = get_items_for_main_table_suggest(query)

    query = _add_filter_sort_direction(args, query)

    logging.info("End `get_aggr_user_cores` query, reqid={reqid}".format(reqid=args["reqid"]))

    cores_table_with_count = put_cores_in_table(query, args[const.MAIN]["p"], tables.TabledMainCore, const.USER_CORES)
    return cores_table_with_count, suggest_items, warnings


def get_core(core_hash, core_itype):
    return models.Core.query.filter(
        models.Core.itype == core_itype,
        models.Core.core_hash == core_hash,
    ).first()


def get_aggr_cores(app_flask, args):
    logging.info('Start `get_aggr_cores` query, reqid=%s', args["reqid"])
    query, warnings = _build_filter_query(args)
    if args[const.FILTER] == const.args_default[const.FILTER]:
        logging.debug("Got default args")
        if not app_flask.suggest_cache:
            app_flask.suggest_cache.update(get_items_for_main_table_suggest(query))
        suggest_items = app_flask.suggest_cache
    else:
        suggest_items = get_items_for_main_table_suggest(query)

    query = _add_filter_sort_direction(args, query)
    logging.info('End `get_aggr_cores` query, reqid={reqid}'.format(reqid=args["reqid"]))
    cores_table_with_count = put_cores_in_table(query, args[const.MAIN]["p"], tables.TabledMainCore, const.INDEX)
    return cores_table_with_count, suggest_items, warnings


def get_ssh_command(instance):
    instance = instance.split(':')[0]
    if "search.yandex.net" in instance:
        return "ssh {}".format(instance)
    else:
        return "ssh {}.search.yandex.net".format(instance)


def get_aggr_cores_single(args):
    logging.debug("Start aggr_cores_single query %s, reqid=%s", datetime.datetime.now(), args["reqid"])
    query = models.CoreDetails.query.filter(
        models.CoreDetails.core_hash == args[const.SINGLE]["core_hash"],
        models.CoreDetails.itype == args[const.SINGLE]["itype"],
    )
    whole_cores_num = query.count()
    logging.debug("Aggr_cores_single after count %s, reqid=%s", datetime.datetime.now(), args["reqid"])
    suggest_items = suggest.get_unique_items(query, ["ctype", "prj", "instance", "tags"], models.CoreDetails)
    logging.debug("Aggr_cores_single after distinct select query {time}".format(time=datetime.datetime.now()))

    server = ".search.yandex.net"
    suggest_items["instance"] = [
        instance[:instance.find(server)] if server in instance else instance for instance in suggest_items["instance"]
    ]
    logging.debug(
        "Ctypes " + ', '.join(map(str, suggest_items["ctype"])) + "\n" +
        "Prjs " + ', '.join(map(str, suggest_items["prj"])) + "\n" +
        "Instances " + ', '.join(map(str, suggest_items["instance"])) + "\n" +
        "Tags " + ", ".join(map(str, suggest_items["tags"]))
    )

    if args[const.SINGLE]["instance"]:
        query = query.filter(models.CoreDetails.instance.startswith(args[const.SINGLE]["instance"]))
    if args[const.SINGLE]["period"]:
        period = args[const.SINGLE]["period"]
        curr_time = time.time()
        if period == "today":
            query = query.filter(models.CoreDetails.timestamp >= int(curr_time) - 86400)  # One day
        elif period == "week":
            query = query.filter(models.CoreDetails.timestamp >= int(curr_time) - 604800)  # One week
    if args[const.SINGLE]["first_time"]:
        first_time_timestamp = time.mktime(parser.parse(args[const.SINGLE]["first_time"]).timetuple())
        query = query.filter(models.CoreDetails.timestamp >= first_time_timestamp)
    if args[const.SINGLE]["last_time"]:
        last_time_timestamp = time.mktime(parser.parse(args[const.SINGLE]["last_time"]).timetuple())
        query = query.filter(models.CoreDetails.timestamp <= last_time_timestamp)
    if args[const.SINGLE]["tags"]:
        tags = get_tags(args[const.SINGLE]["tags"])
        query = query.filter(
            models.CoreDetails.core_id.in_(
                filters.get_core_ids_for_tags_filter(
                    tags, request_tags_argument=args[const.SINGLE]["tags"],
                )
            )
        )

    if args[const.SINGLE]["ctype"]:
        query = query.filter(models.CoreDetails.ctype == args[const.SINGLE]["ctype"])
    if args[const.SINGLE]["prj"]:
        query = query.filter(models.CoreDetails.prj == args[const.SINGLE]["prj"])
    if args[const.SINGLE]["direction"] == "asc":
        query = query.order_by(MODEL[args[const.SINGLE]["sort"]])
    else:
        query = query.order_by(MODEL[args[const.SINGLE]["sort"]].desc())
    logging.debug("Aggr_cores_single after query {time}, reqid={reqid}".format(
        reqid=args["reqid"],
        time=datetime.datetime.now()),
    )
    return put_cores_in_table(query, args[const.SINGLE]["p"], tables.TabledSingleCore), whole_cores_num, suggest_items


def prepare_full_core(core_id):
    # Slow in case of large cores (see CORES-171)
    core = models.CoreSingle.query.filter(models.CoreSingle.core_id == core_id).first()
    if core.error:
        return core.full_trace, True, core.additional_text

    stack_groups = coredump_filter.deserialize_stacks(core.full_trace)
    return stack_groups, False, core.additional_text


def stacks_to_html(stacks):
    ans = ""
    for cur_hash_stacks in stacks:
        same_hash = False
        for stack in cur_hash_stacks:
            ans += stack.html(same_hash=same_hash, same_count=len(cur_hash_stacks), return_result=True)
            same_hash = True
    return ans


def create_link_to_core(core_id):
    return const.DEFAULT_HOST + const.CORE_TRACE + "?core_id={core_id}".format(core_id=core_id)


def create_link_to_cores_with_hash(core_hash, itype):
    return (
        const.DEFAULT_HOST + const.CORE
        + "?core_hash={core_hash}&itype={itype}".format(core_hash=core_hash, itype=itype)
    )


def get_default_query_attrs():
    query_attrs = {
        const.FILTER: const.args_default[const.FILTER],
        const.MAIN: const.args_default[const.MAIN],
    }
    return copy.deepcopy(query_attrs)


def get_user_page_attrs(user_name, page_name):
    query = models.UserRequestAttributes.query.filter(
        models.UserRequestAttributes.user_name == user_name,
        models.UserRequestAttributes.page_name == page_name,
    ).first()
    if not query:
        query_attrs = get_default_query_attrs()
        db.session.add(models.UserRequestAttributes(
            user_name=user_name,
            page_name=page_name,
            attrs=query_attrs,
        ))
        db.session.commit()
    else:
        logging.info("Got query params for user %s: %s", user_name, query.attrs)
        query_attrs = query.attrs

    return query_attrs


def update_user_page_attrs(user_name, page_name, attrs):
    ura = models.UserRequestAttributes.query.filter(
        models.UserRequestAttributes.user_name == user_name,
        models.UserRequestAttributes.page_name == page_name,
    ).first()

    if ura:
        logging.info(
            '[update_user_page_attrs]: updating existing record for user `%s` and page `%s` with attrs: %s',
            user_name, page_name, attrs
        )
        ura.attrs = attrs
    else:
        logging.info(
            '[update_user_page_attrs]: creating new record for user `%s` and page `%s` with attrs: %s',
            user_name, page_name, attrs
        )
        ura = models.UserRequestAttributes(
            user_name=user_name,
            page_name=page_name,
            attrs=attrs,
        )
        db.session.add(ura)

    db.session.commit()
    logging.info('[update_user_page_attrs] done')


def _prepare_args(args):
    if "service" in args:
        args["itype"] = args["service"]
        del args["service"]
    if "server" in args:
        args["instance"] = args["server"]
        del args["server"]
    if "task_id" in args:
        args["sb_task_run"] = args["task_id"]
        del args["task_id"]
    return args
