import os
import time
import logging

from infra.cores.app import models
from infra.cores.app import const
from release_machine.common_proto import events_pb2, notifications_pb2
from sandbox.projects.release_machine import client


def prepare_staff_logins(staff_logins):
    if not staff_logins:
        return {"telegram": [], "q": [], "mail": []}
    entities = [entity.split(":") for entity in staff_logins.split()]
    telegram_chat_names = []
    telegram_chat_ids = []
    q_chat_names = []
    q_chat_ids = []
    mail_addresses = []
    rm_client = client.RMClient()
    for entity_name, notification_types in entities:
        if "telegram" in notification_types:
            telegram_chat_names.append(entity_name + "::telegram")
        if "q" in notification_types:
            q_chat_names.append(entity_name + "::q_messenger")
        if "mail" in notification_types:
            mail_addresses.append(entity_name)
    # Add get request for telegram_chat_ids RMDEV-1452
    logging.debug("START")
    response = rm_client.get_component_chats(
        component_name="cores",
    )
    logging.debug("Got response from rm_client: {}".format(response))
    all_chats = response["chats"] + response["otherChats"]
    chat_name_to_chat_id = {}
    for chat in all_chats:
        if "name" in chat and "chatId" in chat:
            chat_name_to_chat_id[chat["name"]] = chat["chatId"]
    for chat_name in telegram_chat_names:
        if chat_name in chat_name_to_chat_id:
            telegram_chat_ids.append(chat_name_to_chat_id[chat_name])
    for chat_name in q_chat_names:
        if chat_name in chat_name_to_chat_id:
            q_chat_ids.append(chat_name_to_chat_id[chat_name])
    return {"telegram": telegram_chat_ids, "q": q_chat_ids, "mail": mail_addresses}


def prepare_notification_for_q(message, chat_ids):
    return [notifications_pb2.NotificationBundle(
        q_messenger_data=notifications_pb2.QMessengerEventNotificationData(
            chat_id=chat_id,
            message=message,
        ),
    ) for chat_id in chat_ids]


def send_notifications(core_id, itype, instance, staff_logins, core_hash):
    is_empty = core_hash == 0
    prepared_chat_ids = prepare_staff_logins(staff_logins)
    logging.debug("Got prepared_chat_ids: %s", prepared_chat_ids)
    chat_notification_config = const.CHATS.get(itype)
    chat_id_by_itype = getattr(chat_notification_config, "chat_id") if chat_notification_config else None

    last_cores = models.CoreDetails.query.filter(
        models.CoreDetails.core_hash == core_hash,
        models.CoreDetails.itype == itype,
    ).order_by(models.CoreDetails.timestamp.desc()).limit(const.MAX_SAME_CORES_PER_HOUR * 4)

    if last_cores.count() >= const.MAX_SAME_CORES_PER_HOUR * 4:
        # Whether there are more than const.MAX_SAME_CORES_PER_HOUR cores with same hash
        if last_cores[0].timestamp - last_cores[-1].timestamp <= const.SAME_CORES_INTERVAL:
            logging.debug(
                "There are more than %s cores with same hash at last 4 hours, don't send notifications",
                const.MAX_SAME_CORES_PER_HOUR * 4,
            )
            return

    if chat_notification_config and chat_notification_config.only_new_cores and last_cores.count() > 1:
        logging.debug(
            "Get not less than %s cores with flag only_new_cores enabled, don't send notifications", last_cores.count()
        )
        return
    tg_chat_ids = prepared_chat_ids["telegram"]
    tg_chat_ids.append(chat_id_by_itype)
    tg_chat_ids = list(set([chat_id for chat_id in tg_chat_ids if chat_id]))
    if is_empty:
        message = "[{itype}] Got empty core from {instance} ".format(
            itype=itype,
            instance=instance,
        )
    else:
        message = "[{itype}] Instance {instance} core dumped. ".format(
            itype=itype,
            instance=instance,
        )
    q_message = message + "[Details]({host}core_trace?core_id={core_id})".format(
        host=os.getenv("HOST", const.DEFAULT_HOST),
        core_id=core_id,
    )
    message += "<a href='{host}core_trace?core_id={core_id}'>Details</a>".format(
        host=os.getenv("HOST", const.DEFAULT_HOST),
        core_id=core_id,
    )
    logging.debug("Message {msg}, tg_chat_ids {tg_chat_ids}".format(msg=message, tg_chat_ids=tg_chat_ids))
    custom_notifications = prepare_notification_for_telegram(message, tg_chat_ids)
    custom_notifications.extend(prepare_notification_for_q(q_message, prepared_chat_ids["q"]))
    custom_notifications.extend(prepare_notification_for_email(message, prepared_chat_ids["mail"], itype))
    if custom_notifications:
        try:
            event = events_pb2.EventData(
                general_data=events_pb2.EventGeneralData(
                    hash=str(hash(str(core_id) + itype + str(time.time()))),
                    component_name="cores",
                    referrer="cores",
                ),
                new_core_data=events_pb2.NewCoreData(
                    core_id=core_id,
                ),
                custom_notifications=custom_notifications,
            )
            rm_client = client.RMClient()
            rm_client.post_proto_events([event, ])
        except Exception as exc:
            logging.debug("Got exception while sending event to RMClient, exc: %s", exc)


def prepare_message_for_sending(message, chat_ids):
    messages = []
    for chat_id in chat_ids:
        messages.append({
            "chat_id": str(chat_id),
            "message": message.encode("utf-8"),
        })
    return messages


def prepare_emails_for_sending(message, users, itype):
    messages = []
    for user in users:
        messages.append({
            "addr": user + "@yandex-team.ru",
            "subject": "New core for {itype}".format(itype=itype),
            "body": ("New core for {itype}\n{message}".format(itype=itype, message=message)).encode("utf-8"),
        })
    return messages


def prepare_notification_for_email(message, users, itype):
    emails = prepare_emails_for_sending(message, users, itype)
    return [notifications_pb2.NotificationBundle(
        email_data=notifications_pb2.EmailEventNotificationData(
            **email
        ),
    ) for email in emails]


def prepare_notification_for_telegram(message, chat_ids):
    return [notifications_pb2.NotificationBundle(
        telegram_data=notifications_pb2.TelegramEventNotificationData(
            chat_id=str(chat_id),
            message=message,
        ),
    ) for chat_id in chat_ids]
