import os
import logging

from library.python.vault_client.instances import Production as VaultClient
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper

from infra.cores.app import const
from infra.cores.app import db
from infra.cores.app import models
from infra.cores.app import strings
from infra.cores.app import helpers

ST_ROOT = "https://st.yandex-team.ru/"

"""
NOTE about naming: please use:
- `ticket` for `ticket object (not `issue`),
- `ticket_key` for ticket key (not `ticket_id`, etc)
- `issue*` can be used only as ST API internals.
"""


def link_ticket(ticket_key, core_hash, core_itype, core_id=None):
    """
    Link ST ticket to
    :return (ok_status, error_message)
    """
    if ticket_key.startswith(ST_ROOT):
        ticket_key = ticket_key.replace(ST_ROOT, "")

    try:
        ticket = get_ticket(ticket_key)
    except Exception as exc:
        message = "Cannot find ticket `{ticket_key}`.".format(ticket_key=ticket_key)
        logging.exception("Got exception %s while trying to get ticket %s", exc, ticket_key)
        return False, message

    if core_id is not None:
        single_core_query = models.CoreDetails.query.filter(models.CoreDetails.core_id == core_id).first()
        single_core_tickets = single_core_query.tickets
        if ticket.key in strings.parse_tokens(single_core_tickets):
            message = "Core {core_id} is already linked to ticket {ticket_key}".format(
                core_id=core_id,
                ticket_key=ticket.key,
            )
            logging.info("[link_ticket] %s", message)
            return False, message
        single_core_query.tickets = strings.merge_tokens(single_core_tickets, ticket.key)
        single_core_query.expire_time = single_core_query.expire_time + const.SECONDS_IN_MONTH * 6
        db.session.commit()

    aggr_core_query = helpers.get_core(core_hash, core_itype)
    aggr_core_tickets = aggr_core_query.tickets
    if ticket.key in strings.parse_tokens(aggr_core_tickets) and core_id is None:
        message = "Core with hash {core_hash} and itype {core_itype} is already linked to ticket {ticket_key}".format(
            core_hash=core_hash,
            core_itype=core_itype,
            ticket_key=ticket.key,
        )
        logging.debug(message)
        return False, message
    aggr_core_query.tickets = strings.merge_tokens(aggr_core_tickets, ticket.key)
    db.session.commit()

    comment = _create_core_description_for_startrek(
        core_hash=core_hash,
        itype=core_itype,
        core_id=core_id,
    )
    st_helper = get_st_helper()
    st_helper.create_comment(issue=ticket, text=comment)
    return True, ""


def delete_global_link(ticket_key, core_hash, core_itype):
    logging.info('Delete link with {} from aggregate {} : {}'.format(
        ticket_key, core_hash, core_itype))

    aggr_core_query = helpers.get_core(core_hash, core_itype)
    aggr_core_tickets = strings.parse_tokens(aggr_core_query.tickets)
    aggr_core_query.tickets = ",\n".join(
        [ticket for ticket in aggr_core_tickets if ticket != ticket_key]
    )
    db.session.commit()

    cores_details = models.CoreDetails.query.filter(
        models.CoreDetails.itype == core_itype,
        models.CoreDetails.core_hash == core_hash,
        models.CoreDetails.tickets.contains(ticket_key),
    ).all()

    for core_details in cores_details:
        delete_link_from_single_core(ticket_key, core_details.core_id)


def delete_link_from_single_core(ticket_key, core_id):
    logging.info("Destroy link with ticket %s for core %s", ticket_key, core_id)
    single_core_query = models.CoreDetails.query.filter(
        models.CoreDetails.core_id == core_id).first()
    single_core_tickets = strings.parse_tokens(single_core_query.tickets)
    single_core_query.tickets = ",\n".join(
        [ticket for ticket in single_core_tickets if ticket != ticket_key])
    db.session.commit()


def remove_zombie_links(ticket_key, core_hash, core_itype):
    cores_details = models.CoreDetails.query.filter(
        models.CoreDetails.itype == core_itype,
        models.CoreDetails.core_hash == core_hash,
        models.CoreDetails.tickets.contains(ticket_key),
    ).all()

    if not cores_details:
        aggr_core_query = helpers.get_core(core_hash, core_itype)
        aggr_core_tickets = strings.parse_tokens(aggr_core_query.tickets)
        aggr_core_query.tickets = ",\n".join(
            [ticket for ticket in aggr_core_tickets if ticket != ticket_key]
        )
        db.session.commit()


def create_ticket_and_update_core_details(core_id, core_hash, core_itype, assignee):
    ticket = _create_ticket(core_hash, core_itype, assignee, core_id)
    if core_id is not None:
        single_core_query = models.CoreDetails.query.filter(models.CoreDetails.core_id == core_id).first()
        single_core_query.tickets = strings.merge_tokens(single_core_query.tickets, ticket.key)
        single_core_query.expire_time = single_core_query.expire_time + const.SECONDS_IN_MONTH * 6

    core = helpers.get_core(core_hash, core_itype)
    core.tickets = strings.merge_tokens(core.tickets, ticket.key)
    # db.session.add(core)  # FIXME
    db.session.commit()
    bug_report_link = "{}{}".format(ST_ROOT, ticket.key)
    return bug_report_link


def _create_core_description_for_startrek(core_hash, itype, core_id=None):
    same_cores_link = helpers.create_link_to_cores_with_hash(core_hash, itype)

    descr = ""
    if core_id is not None:
        core_link = helpers.create_link_to_core(core_id)
        descr += "(({} Link to core))\n".format(core_link)
    descr += "(({} Link to cores with same hash))\n".format(same_cores_link)
    if core_id is not None:
        query = models.CoreDetails.query.filter(models.CoreDetails.core_id == core_id).first()
        descr += "Instance {instance}\n".format(instance=query.instance)
        full_core, error, additional_text = helpers.prepare_full_core(core_id=core_id)
        if additional_text:
            descr += "Additional info: {}\n".format(additional_text)
        if error:
            logging.info("Problem stack " + str(full_core))
            descr += "<{{Problem stack\n <# {problem_stack} #> }}>\n".format(problem_stack=full_core)
            return descr
        problem_stack = list(full_core)[0]
        descr += "<{{Problem stack\n <# {problem_stack} #> }}>\n".format(
            problem_stack=helpers.stacks_to_html([problem_stack])
        )
    else:
        problem_stack = helpers.get_core(core_hash, itype).core_summary
        descr += "<{{Problem stack\n <# {problem_stack} #> }}>\n".format(problem_stack=problem_stack)
    return descr


def _create_ticket(core_hash, core_itype, assignee, core_id=None):
    """
    :return ST ticket `issue` object
    """
    st_helper = get_st_helper()
    descr = _create_core_description_for_startrek(core_hash=core_hash, itype=core_itype, core_id=core_id)
    queue_for_core = const.STARTREK_QUEUES.get(core_itype, "SEARCH")
    if core_id is None:
        # Needs when ticket is created from page /core without core_id
        last_core_id = helpers.get_core(core_hash, core_itype).last_core_id
        single_core_query = models.CoreDetails.query.filter(models.CoreDetails.core_id == last_core_id).first()
    else:
        single_core_query = models.CoreDetails.query.filter(models.CoreDetails.core_id == core_id).first()
    if single_core_query is not None and single_core_query.ctype == "SDC":
        queue_for_core = "SDC"
    logging.debug("DESCR " + descr)
    kwargs = {
        "queue": queue_for_core,
        "assignee": assignee,
        "summary": "Core from coredumps aggr, core_id {core_id}, core_hash {core_hash}, itype {core_itype}".format(
            core_id=core_id,
            core_hash=core_hash,
            core_itype=core_itype,
        ),
        "description": descr,
    }
    ticket = st_helper.st_client.issues.create(**kwargs)
    return ticket


def get_st_token():
    token = os.getenv("STARTREK_TOKEN")
    if token:
        return token

    yav_token = os.getenv("YAV_TOKEN")
    if not yav_token:
        raise Exception("Both `STARTREK_TOKEN` and `YAV_TOKEN` missing in environment. ")

    # Get token from YAV
    client = VaultClient(
        authorization='OAuth {}'.format(yav_token),
        decode_files=True,
    )
    token = client.get_version(const.VAULT_VERSION)['value']['STARTREK_TOKEN']
    return token


def get_st_helper():
    return STHelper(token=get_st_token(), useragent="robot-srch-releaser")


def get_ticket(ticket_key):
    """
    :return: ST `issue` object
    """
    st_helper = get_st_helper()
    ticket = st_helper.get_ticket_by_key(ticket_key)
    return ticket
