# -*- coding: utf-8 -*-
import logging
import datetime

from flask_table import Table, Col
from flask_table.html import element
from flask import Markup, url_for, render_template

from infra.cores.app import cgi_args
from infra.cores.app import tag_parser
from infra.cores.app import const
from infra.cores.app import strings


class TableWithSearch(Table):
    def __init__(
            self, items, classes=None, thead_classes=None,
            sort_by=None, sort_reverse=False, no_items=None,
            table_id=None, border=None, html_attrs=None, args=None,
    ):
        super(TableWithSearch, self).__init__(
            items, classes, thead_classes, sort_by, sort_reverse, no_items, table_id, border, html_attrs,
        )
        self.args = args

    def th(self, col_key, col):
        if hasattr(col, "_input_field") and col._input_field:
            input_field = render_template(
                "suggest_input_field.html",
                name=col_key,
                value=self.args[const.FILTER][col_key],
                id=col_key,
            )
            return element(
                'th',
                content=self.th_contents(col_key, col) + input_field,
                escape_content=False,
                attrs=col.th_html_attrs,
            )
        else:
            return element(
                'th',
                content=self.th_contents(col_key, col),
                escape_content=False,
                attrs=col.th_html_attrs,
            )


class ColWithInputField(Col):
    def __init__(self, input_field=None, *args, **kwargs):
        super(ColWithInputField, self).__init__(*args, **kwargs)
        self._input_field = input_field


class ColWithoutEscape(ColWithInputField):
    def td_contents(self, i, attr_list):
        return self.from_attr_list(i, attr_list)


class DateCol(ColWithInputField):
    def td_contents(self, i, attr_list):
        return Markup.escape(
            datetime.datetime.fromtimestamp(
                int(self.from_attr_list(i, attr_list))
            ).strftime('%X\n%d.%m.%Y')
        )


class LimitedHeightCol(ColWithInputField):
    def td_contents(self, i, attrs_list):
        return "<div style='max-height: 120px; overflow: auto'>" + self.from_attr_list(i, attrs_list) + "</div>"


class MainTable(TableWithSearch):
    def __init__(self, elements, sort_by, sort_reverse, classes, thead_classes, page, args):
        self.page = page
        super(MainTable, self).__init__(
            elements,
            sort_by=sort_by,
            sort_reverse=sort_reverse,
            classes=classes,
            thead_classes=thead_classes,
            args=args,
        )

    first_time = DateCol(input_field=False, name="First time", column_html_attrs={"class": "size_date"})
    last_time = DateCol(input_field=False, name="Last time", column_html_attrs={"class": "size_date"})
    itype = LimitedHeightCol(input_field=False, name="Itype", column_html_attrs={"class": "size_service"})
    ctype_list = LimitedHeightCol(
        input_field=False,
        name="Cluster Type",
        column_html_attrs={
            "class": "size_ctype",
            "title": "Cluster type, also known as `ctype` tag, e.g. `production`, `prestable`, `test`, `hamster`, etc",
        })
    prj_list = LimitedHeightCol(input_field=False, name="Prj", column_html_attrs={"class": "size_prj"})
    tag_list = LimitedHeightCol(
        input_field=False, name="Tags", column_html_attrs={"class": "size_tags"}, allow_sort=False,
    )
    host_list = LimitedHeightCol(
        input_field=False, name="Hosts", column_html_attrs={"class": "size_host"}, allow_sort=False,
    )
    trace = ColWithoutEscape(input_field=False, name="Trace", allow_sort=False)
    signal = ColWithInputField(input_field=False, name="Signal", column_html_attrs={"class": "size_signal"})
    count = ColWithInputField(
        input_field=False,
        name="Count",
        column_html_attrs={"class": "size_count"},
        td_html_attrs={"align": "left"},
    )
    links = ColWithoutEscape(name="Details", column_html_attrs={"class": "size_details"}, allow_sort=False)
    allow_sort = True

    def sort_url(self, col_key, reverse=False):
        direction = "desc" if reverse else "asc"
        return cgi_args.custom_url_for(
            cur_cat=const.MAIN,
            args_new=self.args,
            page=self.page,
            sort=col_key,
            direction=direction,
            p=1,
        )


class TabledMainCore(object):

    def __init__(self, db_core, page_name):
        self.first_time = db_core.first_time
        self.last_time = db_core.last_time
        self.itype = db_core.itype
        self.ctype_list = db_core.ctype_list
        self.prj_list = db_core.prj_list
        self.host_list = db_core.host_list
        self.tag_list = db_core.tag_list
        self.trace = db_core.core_summary
        self.signal = "{sig_name}({sig_number})".format(
            sig_name=const.SIG_NUM_TO_SIG_NAME.get(db_core.signal, ""),
            sig_number=db_core.signal,
        )
        self.count = db_core.count
        self.core_hash = db_core.core_hash
        self.core_fixed = db_core.fixed
        self.last_core_id = db_core.last_core_id
        self.page_name = page_name
        self.tickets = db_core.tickets
        self.links = self.prepare_core_links()

    def prepare_core_links(self):
        core_link = url_for(const.CORE, core_hash=self.core_hash, itype=self.itype)
        ticket_keys = strings.parse_tokens(self.tickets)
        ticket_keys = list(set(ticket_keys))
        fixed_link = url_for(const.MARK_AS_FIXED, core_hash=self.core_hash, itype=self.itype, page_name=self.page_name)
        last_core_link = url_for(const.CORE_TRACE, core_id=self.last_core_id)
        return render_template(
            "core_links.html",
            core_link=core_link,
            fixed_link=fixed_link,
            core_fixed=self.core_fixed,
            last_core_link=last_core_link,
            ticket_keys=ticket_keys,
        )


class TabledSingleCore(object):
    def __init__(self, db_core):
        self.timestamp = db_core.timestamp
        self.instance = db_core.instance
        self.core_hash = db_core.core_hash
        self.full_trace = db_core.top_frame
        self.details = db_core.top_frame
        self.ticket = 1
        self.sb_build = db_core.sb_build
        self.sb_task_run = db_core.sb_task_run
        self.itype = db_core.itype
        self.ctype = db_core.ctype
        self.signal = db_core.signal
        self.prj = db_core.prj
        self.tags = db_core.tags
        self.core_id = db_core.core_id
        self.tickets = db_core.tickets
        self.core_info = self.prepare_core_info()

    def period_format(self, periods_count, period_name):
        if periods_count > 0:
            output_str = "{periods_count} {period_name}".format(periods_count=periods_count, period_name=period_name)
            if periods_count > 1:
                output_str += "s"
            return output_str
        return ""

    def prepare_core_info(self):
        core_id = self.core_id
        tickets = strings.parse_tokens(self.tickets)
        link = url_for(const.CORE_TRACE, core_id=core_id)
        tags = tag_parser.get_rendered_tags(self.tags)

        logging.debug("Core_id " + str(core_id))
        seconds_ago = int(datetime.datetime.now().strftime('%s')) - self.timestamp
        years_ago = self.period_format(seconds_ago // const.SECONDS_IN_YEAR, "year")
        months_ago = self.period_format(seconds_ago // const.SECONDS_IN_MONTH, "month")
        days_ago = self.period_format(seconds_ago // const.SECONDS_IN_DAY, "day")
        hours_ago = self.period_format(seconds_ago // const.SECONDS_IN_HOUR, "hour")
        minutes_ago = self.period_format(seconds_ago // const.SECONDS_IN_MINUTE, "minute")
        seconds_ago = self.period_format(seconds_ago, "second")
        time_ago = years_ago or months_ago or days_ago or hours_ago or minutes_ago or seconds_ago

        return render_template(
            "tabled_single_core.html",
            link=link,
            itype=self.itype,
            prj=self.prj,
            ctype=self.ctype,
            tags=tags,
            signal=self.signal,
            time=datetime.datetime.fromtimestamp(self.timestamp).strftime('%Y.%m.%d\n%X'),
            time_ago=time_ago,
            instance=self.instance,
            core_id=self.core_id,
            core_hash=self.core_hash,
            sb_build=self.sb_build if self.sb_build else "",
            sb_task_run_link=url_for(const.SANDBOX, sb_task_run=self.sb_task_run) if self.sb_task_run else "",
            sb_task_run=self.sb_task_run if self.sb_task_run else "",
            ticket_link=url_for(const.CREATE_TICKET, core_id=core_id, core_hash=self.core_hash, core_itype=self.itype),
            tickets=tickets,
        )

    @staticmethod
    def normalize(core_list):
        return map(lambda x: TabledSingleCore(x), core_list)


class SingleCoreTable(TableWithSearch):
    full_trace = ColWithoutEscape(input_field=False, name="Full", allow_sort=False)
    core_info = ColWithoutEscape(name="Core info", column_html_attrs={"class": "size_core_info"}, allow_sort=False)

    allow_sort = True

    def sort_url(self, col_key, reverse=False):
        direction = "desc" if reverse else "asc"
        return cgi_args.custom_url_for(
            cur_cat=const.SINGLE,
            args_new=self.args,
            page=const.CORE,
            sort=col_key,
            direction=direction,
            p=1,
        )
