from infra.cores.app import strings

STATVIEWER_URL = "https://simulator.sdc.yandex-team.ru/stats"
SDC_WWW_RIDES = "https://sdc.yandex-team.ru/rides"
TEAMCITY_URL = "https://teamcity.yandex-team.ru"


class TagRender(object):
    def __init__(self, tag_name):
        self.tags_processor_mapper = {
            'ride': self._ride_url,
            'teamcity_build_id': self._teamcity_url,
            'statviewer_build_id': self._statviewer_url,
            'named_link': self._named_link,
        }
        self.text = None
        self.title = None
        self.tag_type = None
        self.tag_name = tag_name

    def render(self, content):
        processor = self.tags_processor_mapper.get(self.tag_name, self._default_processor)
        processor(content)

        return self.text

    def _ride_url(self, ride):
        self.text = '{}/{}'.format(SDC_WWW_RIDES, ride)
        self.tag_type = 'url'
        self.title = ride

    def _statviewer_url(self, parent_build_id):
        self.text = '{base_url}?parent_build_id={id}'.format(
            base_url=STATVIEWER_URL,
            id=parent_build_id,
        )
        self.tag_type = 'url'
        self.title = parent_build_id

    def _teamcity_url(self, teamcity_build_id):
        self.text = '{base_url}/viewLog.html?buildId={id}'.format(
            base_url=TEAMCITY_URL,
            id=teamcity_build_id,
        )
        self.tag_type = 'url'
        self.title = teamcity_build_id

    def _named_link(self, content):
        parts = content.split(':')
        self.text = ':'.join(parts[:-1])
        self.title = parts[-1]
        self.tag_type = 'url'

    def _default_processor(self, content):
        self.text = content
        self.title = content
        self.tag_type = 'plain-text'


def _named_tag(content, tag_name='tag_name'):
    render = TagRender(tag_name)
    render.render(content)

    return render


def tags_parser(tag):
    if not tag:
        return None

    parts = tag.split(':', 1)
    if len(parts) == 2:
        name, content = parts
        return _named_tag(tag_name=name, content=content)

    return _named_tag(content=tag)


def get_rendered_tags(tags):
    if not tags:
        return []

    printable_tags = []
    for tag in strings.parse_tokens(tags):
        formatted_tag = tags_parser(tag)
        if formatted_tag:
            printable_tags.append(formatted_tag)

    return printable_tags
