# -*- coding: utf-8 -*-

import html
import json
import copy
import datetime
import logging
import random
import socket
import string

import flask as fl
from infra.cores.app import cgi_args
from infra.cores.app import date_time
from infra.cores.app import db
from infra.cores.app import garbage_collector
from infra.cores.app import background_thread
from infra.cores.app import helpers
from infra.cores.app import tables
from infra.cores.app import tag_parser
from infra.cores.app import models
from infra.cores.app import strings
from infra.cores.app import st_tickets
from infra.cores.app import const
from infra.cores.app import view_submit_core
from infra.cores.app import view_core_info
from infra.cores.app import view_core_details
from infra.cores.app import view_core_list_ids


def _page(page_name):
    return "/{}".format(page_name)


def init_routes(app_flask):

    @app_flask.before_request
    def log_request_info():
        letters = string.ascii_lowercase
        fl.request.reqid = "".join(
            [random.choice(letters) for i in range(5)]
        ) + str(random.randint(10 ** 5, 5 * 10 ** 5))
        logging.info('Request: %s, ReqID:%s', fl.request, fl.request.reqid)

    @app_flask.after_request
    def after_request(response):
        logging.info(
            '%s %s %s %s %s',
            fl.request.remote_addr,
            fl.request.method,
            fl.request.scheme,
            fl.request.full_path,
            response.status,
        )
        return response

    @app_flask.teardown_appcontext
    def shutdown_session(exception=None):
        db.session.remove()

    @app_flask.route("/get_my_ip", methods=["GET"])
    def get_my_ip():
        login = fl.request.cookies.get("yandex_login")
        ans = {"login": login}
        return _response_to_json(ans)

    @app_flask.route("/slow_code", methods=["GET"])
    def slow_code():
        logging.error("PROFILE_BEGIN")
        core_id = int(fl.request.args.get("core_id", 39288757))
        import cProfile
        import pstats
        with cProfile.Profile() as pr:
            core = models.CoreSingle.query.filter(models.CoreSingle.core_id == core_id).first()
            print(core)
        pr.print_stats(sort=pstats.SortKey.CUMULATIVE)
        logging.error("PROFILE_END")
        return str(core) + " :: " + str(core_id)

    @app_flask.route("/background_tick")
    def background_tick():
        """For background thread functional tests: invoke GC tick"""
        r_args = _get_request_args()
        invocation_time = r_args.get("invocation_time", "")
        continue_flag = False
        exception_text = None

        if not invocation_time.isdigit():
            # show some hint on incorrect usage
            exception_text = 'Usage: /background_tick?invocation_time=<number>'
        else:
            continue_flag, exception_text = background_thread.background_thread_tick(
                app_flask, int(invocation_time),
            )

        return '{}\n\n{}'.format(
            'CONTINUE' if continue_flag else 'STOP',
            'ERROR:\n' + exception_text if exception_text is not None else '',
        )

    @app_flask.route(_page(const.MAIN_URL))
    @app_flask.route(_page(const.INDEX))   # TODO: drop it, leave '/' only
    def index():
        r_args = _get_request_args()
        user_name = helpers.get_user_name(fl.request)
        user_page_attrs = helpers.get_user_page_attrs(user_name, const.INDEX)
        if not r_args:
            return fl.redirect(cgi_args.custom_url_for(cur_cat=const.MAIN, args_new=user_page_attrs, page=const.INDEX))
        else:
            args = cgi_args.update_dict(
                copy.deepcopy(const.args_default),
                r_args,
                cur_cat=const.MAIN,
            )
            helpers.update_user_page_attrs(user_name, const.INDEX, args)

        args["reqid"] = fl.request.reqid
        sort = r_args.get('sort', 'last_time')
        reverse = (r_args.get('direction', 'asc') == 'desc')
        big_table_with_count, suggest_items, warnings = helpers.get_aggr_cores(app_flask, args)
        big_table, max_cores = big_table_with_count
        filter_values = {
            "itype": r_args.get("itype", ""),
            "ctype_list": r_args.get("ctype_list", ""),
            "prj_list": r_args.get("prj_list", ""),
            "host_list": r_args.get("host_list", ""),
            "period": r_args.get("period", ""),
            "first_time": r_args.get("first_time", ""),
            "last_time": r_args.get("last_time", ""),
            "signal": r_args.get("signal", ""),
            "tag_list": r_args.get("tag_list", ""),
        }
        if max_cores is None:
            table = u'Корок с такими параметрами запроса сейчас в базе нет, попробуйте изменить их'
            max_p = 1
            logging.info("Got no cores")
        else:
            table = tables.MainTable(
                big_table,
                sort_by=sort,
                sort_reverse=reverse,
                classes=["table-striped table table-hover table-bordered"],
                thead_classes=["thead-dark"],
                page=const.INDEX,
                args=args,
            )
            max_p = int(max_cores / const.LIMIT_PER_PAGE) + int(max_cores % const.LIMIT_PER_PAGE > 0)
        hostname = socket.gethostname()
        return fl.render_template(
            "main_table.html",
            content=u"Три корочки",
            reqid=fl.request.reqid,
            hostname=hostname,
            table=table,
            page=const.INDEX,
            filter_values=filter_values,
            text_to_search=args[const.FILTER].get("text_to_search", ""),
            show_fixed=args[const.FILTER].get("show_fixed", False),
            suggest_items=json.dumps(suggest_items),
            filtered_cores_num=max_cores,
            p=int(args[const.MAIN]["p"]),
            max_p=max_p,
            test_func=lambda x: cgi_args.custom_url_for(const.MAIN, args, page=const.INDEX, p=x),
            url_func=lambda: cgi_args.custom_url_for(const.MAIN, args, page=const.INDEX),
            warnings=warnings,
        )

    @app_flask.route(_page(const.CLEAR_ALL_FILTERS))
    def clear_all_filters():
        r_args = _get_request_args()
        args = helpers.get_default_query_attrs()
        args["reqid"] = fl.request.reqid
        page = r_args.get("page")
        if page == "/":
            page = const.INDEX
        cur_cat = const.MAIN
        return fl.redirect(cgi_args.custom_url_for(cur_cat=cur_cat, args_new=args, page=page))

    @app_flask.route(_page(const.CORE), methods=["GET", "POST"])
    def core():
        r_args = _get_request_args()
        args = copy.deepcopy(const.args_default)
        args = copy.deepcopy(cgi_args.update_dict(args, r_args, const.SINGLE))
        args["reqid"] = fl.request.reqid
        sort = r_args.get('sort', 'date')
        reverse = (r_args.get('direction', 'asc') == 'desc')
        big_table, whole_cores_num, suggest_items = helpers.get_aggr_cores_single(args)
        big_table, max_cores = big_table
        filter_cores = {
            "ctype": r_args.get("ctype", ""),
            "prj": r_args.get("prj", ""),
            "instance": r_args.get("instance", ""),
            "period": r_args.get("period", ""),
            "first_time": r_args.get("first_time", ""),
            "last_time": r_args.get("last_time", ""),
            "tags": r_args.get("tags", ""),
        }

        if (
            max_cores is None or args["single"].get("itype", None) is None
            or args["single"].get("core_hash", None) is None
        ):
            if args["single"].get("itype", ""):
                table = u'Корок с такими параметрами запроса сейчас в базе нет, попробуйте изменить их'
            else:
                table = u'Укажите itype корки, которую вы ищете. core_hash и itype - обязательные параметры запроса.'
            max_p = 1
        else:
            table = tables.SingleCoreTable(
                big_table,
                sort_by=sort,
                sort_reverse=reverse,
                classes=["table-striped table table-hover table-bordered"],
                thead_classes=["thead-dark"],
                args=args,
            )
            max_p = int(max_cores / const.LIMIT_PER_PAGE) + int(max_cores % const.LIMIT_PER_PAGE > 0)
        hostname = socket.gethostname()
        aggr_core_tickets = helpers.get_core(args["single"].get("core_hash", None), args["single"].get("itype", None))
        ticket_keys = strings.parse_tokens(aggr_core_tickets.tickets)
        return fl.render_template(
            "single_core_table.html",
            content=u"Три корочки",
            reqid=fl.request.reqid,
            hostname=hostname,
            table=table,
            page=const.CORE,
            filter=filter_cores,
            core_hash=r_args.get("core_hash", "1"),
            itype=r_args.get("itype", ""),
            p=int(args[const.SINGLE]["p"]),
            max_p=max_p,
            filtered_cores_num=max_cores,
            whole_cores_num=whole_cores_num,
            suggest_items=json.dumps(suggest_items),
            test_func=lambda x: cgi_args.custom_url_for(const.SINGLE, args, page=const.CORE, p=x),
            url_func=lambda: cgi_args.custom_url_for(const.SINGLE, args, page=const.CORE),
            ticket_keys=ticket_keys,
        )

    @app_flask.route(_page(const.CORE_TRACE))
    def core_trace():
        r_args = _get_request_args()
        core_id = r_args.get("core_id", "1")
        core_details_query = models.CoreDetails.query.filter(models.CoreDetails.core_id == core_id)
        logging.error("CoreDetailsQuery %s", core_details_query)
        if core_details_query.count() == 0:
            return "There is no core with id {}. Please change id to existing one.".format(core_id)

        core_details = core_details_query.first()
        ssh_command = helpers.get_ssh_command(core_details.instance)
        core_tickets = strings.parse_tokens(core_details.tickets)
        ticket_keys = [ticket_key for ticket_key in core_tickets]
        tickets = [st_tickets.get_ticket(ticket_key) for ticket_key in ticket_keys]
        tickets = filter(lambda x: x, tickets)  # filter empty tickets

        full_core, error, additional_text = helpers.prepare_full_core(core_id=core_id)
        expire_time = (core_details.expire_time - date_time.icurrent()) // const.SECONDS_IN_DAY
        additional_text = html.escape(additional_text).replace("\n", "<br/>")
        if error:
            core = full_core
        else:
            core = helpers.stacks_to_html(full_core)
        logging.debug("SSH command: %s", ssh_command)
        hostname = socket.gethostname()
        tags = tag_parser.get_rendered_tags(core_details.tags)

        core_details_info = fl.render_template(
            "tabled_single_trace.html",
            itype=core_details.itype,
            prj=core_details.prj,
            ctype=core_details.ctype,
            tags=tags,
            expire_time=expire_time,
            signal=core_details.signal,
            time=datetime.datetime.fromtimestamp(core_details.timestamp).strftime('%X\n%d.%m.%Y'),
            instance=core_details.instance,
            sb_build=core_details.sb_build if core_details.sb_build else "",
            # FIXME: trash
            sb_task_run_link=(
                fl.url_for(const.SANDBOX, sb_task_run=core_details.sb_task_run)
                if core_details.sb_task_run else ""
            ),
            sb_task_run=core_details.sb_task_run if core_details.sb_task_run else "",
        )
        return fl.render_template(
            "single_trace.html",
            core_details_info=core_details_info,
            reqid=fl.request.reqid,
            hostname=hostname,
            core=core,
            date=datetime.datetime.fromtimestamp(int(core_details.timestamp)).strftime('%d-%m-%Y %H:%M:%S'),
            instance="{}".format(core_details.instance),
            instance_path=ssh_command,
            tickets=tickets,
            core_id=core_id,
            core_hash=core_details.core_hash,
            itype=core_details.itype,
            additional_text=additional_text,
        )

    @app_flask.route(_page(const.CREATE_TICKET), methods=['GET', 'POST'])
    def create_ticket():
        r_args = _get_request_args()
        user_name = helpers.get_user_name(fl.request)
        core_id = r_args.get("core_id")
        core_hash = r_args.get("core_hash")
        core_itype = r_args.get("core_itype")
        logging.info("use core_id: %s for creating ticket", core_id)
        link = st_tickets.create_ticket_and_update_core_details(
            core_id=core_id,
            core_hash=core_hash,
            core_itype=core_itype,
            assignee=user_name,
        )
        return fl.redirect(link)

    @app_flask.route(_page(const.CORE_INFO), methods=['GET'])
    def core_info():
        answer = view_core_info.core_info_impl(r_args=_get_request_args())

        return _response_to_json(answer)

    @app_flask.route(_page(const.CORE_LIST_IDS), methods=['GET'])
    def core_list_ids():
        answer = view_core_list_ids.core_list_ids_impl(r_args=_get_request_args())

        return _response_to_json(answer)

    @app_flask.route(_page(const.CORE_DETAILS), methods=['GET'])
    def core_details():
        answer = view_core_details.core_details_impl(r_args=_get_request_args())

        return _response_to_json(answer)

    @app_flask.route(_page(const.SUBMIT_CORE), methods=['GET', 'POST'])
    def submit_core():
        answer = view_submit_core.submit_core_impl(r_args=_get_request_args())
        status_code = 400 if answer['error'] else 200

        return _response_to_json(answer, status_code=status_code)

    @app_flask.route('/corecomes', methods=['GET', 'POST'])
    def get_cores():
        if fl.request.method == 'GET':
            logging.info('get GET request %s', fl.request)
            return "Need POST request"

        if fl.request.method == "POST":
            logging.info('get POST request %s', fl.request)
            r_args = _get_request_args()

            json_data = fl.request.form
            dump_json = json_data.get('dump_json', {})
            additional_text = json_data.get('additional_text', "")
            logging.debug("Got JSON_data %s", str(json_data))
            logging.debug("Got dump_json %s", str(dump_json))
            logging.debug("Got additional_text %s", str(additional_text))
            properties = dump_json.get("properties", {})
            prj = properties.get("INSTANCE_TAG_PRJ", "")

            args = r_args.to_dict()
            args["prj"] = args.get("prj", "") or prj
            logging.debug("Got args %s", str(args))

            if 'parsed_traces' not in json_data:
                return 'Invalid request, should contain `parsed_traces` in main dict. ', 400

            helpers.add_core(json_data['parsed_traces'], args, additional_text)

        return "All right, sir!", 200

    @app_flask.route(_page(const.ADD_CORE_TYPE), methods=['POST', 'GET'])
    def add_core_type():
        user_name = helpers.get_user_name(fl.request)
        r_args = _get_request_args()
        new_core_itype = r_args.get("core_itype").strip()
        new_core_ctype = r_args.get("core_ctype").strip()
        new_core_prj = r_args.get("core_prj").strip()
        new_core_tag = r_args.get("core_tag").strip()
        core_added = helpers.add_core_type(
            user_name=user_name,
            core_itype=new_core_itype,
            core_ctype=new_core_ctype,
            core_prj=new_core_prj,
            core_tag=new_core_tag,
        )
        if core_added:
            return _redirect(const.USER_SETTINGS)
        else:
            fl.flash("Core type with itype {itype}, ctype {ctype} and prj {prj} is already registered".format(
                itype=new_core_itype,
                ctype=new_core_ctype,
                prj=new_core_prj,
            ))
            return _redirect(const.USER_SETTINGS)

    @app_flask.route(_page(const.EXTEND_CORE_TTL))
    def extend_core_ttl():
        r_args = _get_request_args()
        core_ttl = r_args.get("core_ttl").strip()
        core_id = r_args.get("core_id", "1")
        try:
            core_ttl = int(core_ttl)
            garbage_collector.update_expire_time(core_id, core_ttl)
        except Exception:
            logging.debug("Got incorrect ttl: %s", core_ttl)
            fl.flash("Enter valid integer, not {}".format(core_ttl))
        return _redirect(const.CORE_TRACE, core_id=core_id)

    @app_flask.route(_page(const.MARK_AS_FIXED), methods=['POST', 'GET'])
    def mark_as_fixed():
        r_args = _get_request_args()
        logging.debug("ARGS %s", r_args)
        core_hash = r_args.get("core_hash", "1")
        page_name = r_args.get("page_name", "")
        logging.debug("GOT PAGE_NAME %s", page_name)
        itype = r_args.get("itype", "")
        core = helpers.get_core(core_hash, itype)
        if not core:
            error_msg = "Couldn't find core with hash {hash} and itype {itype}".format(hash=core_hash, itype=itype)
            logging.debug(error_msg)
            fl.flash(error_msg, "danger")
        else:
            core.fixed = not core.fixed
            db.session.commit()
            action = "fixed" if core.fixed else "unfixed"
            success_msg = "Core with hash {hash} and itype {itype} successfully {action}".format(
                hash=core_hash,
                itype=itype,
                action=action,
            )
            logging.debug(success_msg)
            fl.flash(success_msg, "success")
        return _redirect(page_name)

    @app_flask.route(_page(const.LINK_TICKET), methods=['POST', 'GET'])
    def link_ticket():
        if fl.request.method == "GET":
            r_args = _get_request_args()
        elif fl.request.method == "POST":
            r_args = fl.request.get_json() or {}

        logging.debug("Got create_ticket args: %s", r_args)
        ticket_key = r_args.get("ticket_key").strip()
        core_id = r_args.get("core_id", None)
        core_hash = r_args.get("core_hash", "1")
        core_itype = r_args.get("core_itype", "1")
        if ticket_key:
            result, msg = st_tickets.link_ticket(ticket_key, core_hash, core_itype, core_id)
            if not result:
                fl.flash(msg)  # FIXME(mvel) will it work with redirect???
        if core_id is not None:
            return _redirect(const.CORE_TRACE, core_id=core_id)
        else:
            return _redirect(const.CORE, itype=core_itype, core_hash=core_hash)

    @app_flask.route(_page(const.DELETE_TICKET_LINK), methods=['POST', 'GET'])
    def delete_ticket_link():
        if fl.request.method == "GET":
            r_args = _get_request_args()
        elif fl.request.method == "POST":
            r_args = fl.request.get_json() or {}
        logging.debug("Got create_ticket args: %s", r_args)
        ticket_key = r_args.get("ticket_key").strip()
        core_hash = r_args.get("core_hash", "1")
        core_itype = r_args.get("core_itype", "1")
        core_id = r_args.get("core_id", None)

        if core_id is None or core_id == "0":
            st_tickets.delete_global_link(ticket_key, core_hash, core_itype)
        else:
            st_tickets.delete_link_from_single_core(ticket_key, core_id)
            st_tickets.remove_zombie_links(ticket_key, core_hash, core_itype)

        fl.flash("Link to ticket successfully deleted")
        if core_hash == "1":  # TODO(epsilond1)
            return _redirect(const.CORE_TRACE, core_id=core_id)
        else:
            return _redirect(const.CORE, itype=core_itype, core_hash=core_hash)

    @app_flask.route(_page(const.REMOVE_CORE_TYPE))
    def remove_core_type():
        r_args = _get_request_args()
        core_itype = r_args.get("core_itype").strip()
        core_ctype = r_args.get("core_ctype").strip()
        core_prj = r_args.get("core_prj").strip()
        core_tag = r_args.get("core_tag").strip()
        user_name = helpers.get_user_name(fl.request)
        helpers.remove_core_type(
            user_name=user_name,
            core_itype=core_itype,
            core_ctype=core_ctype,
            core_prj=core_prj,
            core_tag=core_tag,
        )
        return _redirect(const.USER_SETTINGS)

    @app_flask.route(_page(const.USER_SETTINGS))
    def user_settings():
        _ = _get_request_args()
        user_name = helpers.get_user_name(fl.request)
        user_cores_itypes = helpers.get_user_cores_types(user_name)
        hostname = socket.gethostname()
        return fl.render_template(
            "user_settings.html",
            hostname=hostname,
            page=const.USER_SETTINGS,
            reqid=fl.request.reqid,
            core_types=user_cores_itypes,
        )

    @app_flask.route(_page(const.USER_CORES))
    def user_cores():
        r_args = _get_request_args()
        user_name = helpers.get_user_name(fl.request)
        if not r_args:
            args = helpers.get_user_page_attrs(user_name, const.USER_CORES)
            return fl.redirect(cgi_args.custom_url_for(cur_cat=const.MAIN, args_new=args, page=const.USER_CORES))
        else:
            args = cgi_args.update_dict(
                copy.deepcopy(const.args_default),
                r_args,
                cur_cat=const.MAIN,
            )
            helpers.update_user_page_attrs(user_name, const.USER_CORES, args)
        args["reqid"] = fl.request.reqid
        user_cores_types = helpers.get_user_cores_types(user_name)

        filter_values = {
            "itype": r_args.get("itype", ""),
            "ctype_list": r_args.get("ctype_list", ""),
            "prj_list": r_args.get("prj_list", ""),
            "host_list": r_args.get("host_list", ""),
            "period": r_args.get("period", ""),
            "first_time": r_args.get("first_time", ""),
            "last_time": r_args.get("last_time", ""),
            "signal": r_args.get("signal", ""),
            "tag_list": r_args.get("tag_list", ""),
        }

        if user_cores_types:
            sort = r_args.get('sort', 'last_time')
            reverse = (r_args.get('direction', 'asc') == 'desc')
            big_table, suggest_items, warnings = helpers.get_aggr_user_cores(args, user_cores_types)
            big_table, max_cores = big_table
            if max_cores is not None:

                table = tables.MainTable(
                    big_table,
                    sort_by=sort,
                    sort_reverse=reverse,
                    classes=["table-striped table table-hover table-bordered"],
                    thead_classes=["thead-dark"],
                    page=const.USER_CORES,
                    args=args,
                )
                hostname = socket.gethostname()
                return fl.render_template(
                    "main_table.html",
                    content=u"Три корочки",
                    reqid=fl.request.reqid,
                    hostname=hostname,
                    table=table,
                    page=const.USER_CORES,
                    suggest_items=json.dumps(suggest_items),
                    filtered_cores_num=max_cores,
                    filter_values=filter_values,
                    p=int(args[const.MAIN]["p"]),
                    text_to_search=args[const.FILTER].get("text_to_search", ""),
                    show_fixed=args[const.FILTER].get("show_fixed", False),
                    max_p=int(max_cores / const.LIMIT_PER_PAGE) + int(max_cores % const.LIMIT_PER_PAGE > 0),
                    test_func=lambda x: cgi_args.custom_url_for(const.MAIN, args, page=const.USER_CORES, p=x),
                    url_func=lambda: cgi_args.custom_url_for(const.MAIN, args, page=const.USER_CORES),
                    core_types=user_cores_types,
                    user_name=user_name,
                    warnings=warnings,
                )

        if user_cores_types:
            message = u'Корок с вашими тегами сейчас в базе нет, попробуйте изменить их или указать новые теги'
        else:
            message = u'Укажите теги корок, которые вы хотели бы видеть'
        hostname = socket.gethostname()
        return fl.render_template(
            "main_table.html",
            content=u"Три корочки",
            reqid=fl.request.reqid,
            hostname=hostname,
            page=const.USER_CORES,
            table=message,
            filtered_cores_num=0,
            filter_values=filter_values,
            p=int(args[const.MAIN]["p"]),
            text_to_search=args[const.FILTER].get("text_to_search", ""),
            show_fixed=args[const.FILTER].get("show_fixed", False),
            max_p=1,
            test_func=lambda x: cgi_args.custom_url_for(const.MAIN, args, page=const.USER_CORES, p=x),
            url_func=lambda: cgi_args.custom_url_for(const.MAIN, args, page=const.USER_CORES),
            core_types=user_cores_types,
            user_name=user_name,
        )

    @app_flask.route(_page(const.SANDBOX))
    def sandbox():
        r_args = _get_request_args()
        link = "https://sandbox.yandex-team.ru/task/{}/view".format(r_args.get("sb_task_run", "1"))
        return fl.redirect(link)

    @app_flask.route(_page('favicon.ico'))
    def favicon():
        """
        CORES-65: Handle favicon properly.
        Please not that `fl.send_from_directory` will not work properly
        because of bundle binary compiling (and `app_flask.root_path`
        points to non-existent folder.
        """
        # _ = _get_request_args()
        return fl.redirect('/static/favicon.ico')

    @app_flask.template_filter('st_ticket')
    def st_ticket(s):
        return "{}{}".format(st_tickets.ST_ROOT, s)


def _redirect(*args, **kwargs):
    return fl.redirect(fl.url_for(*args, **kwargs))


def _get_request_args():
    r_args = fl.request.args
    logging.debug("REQUEST ARGS %s", r_args)
    return r_args


def _response_to_json(payload, status_code=200):
    logging.debug(payload)

    return (
        fl.jsonify(**payload),
        status_code,
        {'Content-Type': 'application/json'}
    )
