#!/usr/bin/env python
from __future__ import print_function
from __future__ import unicode_literals

import argparse
import datetime
import os
import sys
import re

import ujson
import requests

from nanny_rpc_client import requests_client
from clusterpb import federated_stub, federated_pb2
from clusterpb import hq_pb2, hq_stub


PATH_EMIT = '/nanny/services/events_migration/emit_new_sync_events/'
NANNY_URL = 'http://nanny.yandex-team.ru'


def _get_scheduling_policy(s):
    return s['info_attrs']['content'].get('scheduling_policy', {}).get('type', 'NONE')


def _get_summary_state(s):
    return s['current_state']['content']['summary']['value']


def _get_file_resource_ctl_task_id(s):
    for r in s['runtime_attrs']['content']['resources']['sandbox_files']:
        if r['resource_type'] == 'INSTANCECTL':
            return int(r['task_id'])


def _get_ctl_task_id(s):
    for r in s['runtime_attrs']['content']['resources']['sandbox_files']:
        if r['resource_type'] == 'INSTANCECTL':
            return int(r['task_id'])
    spec = s['runtime_attrs']['content'].get('instance_spec', {})
    r = spec.get('instancectl', {}).get('fetchableMeta', {}).get('sandboxResource', {}).get('taskId')
    if r:
        return int(r)
    return None


def _is_juggler_enabled(s):
    c = s['runtime_attrs']['content'].get('instance_spec', {}).get('auxDaemons', [])
    return {"type": "JUGGLER_AGENT"} in c


def _is_skynet_enabled(s):
    c = s['runtime_attrs']['content'].get('instance_spec', {}).get('auxDaemons', [])
    return {"type": "SKYNET"} in c


def _is_yasm_subagent_enabled(s):
    c = s['runtime_attrs']['content'].get('instance_spec', {}).get('hostProvidedDaemons', [])
    return {"type": "YASM_AGENT"} in c


def _has_dyn_resources(s):
    c = ujson.dumps(s['runtime_attrs']['content'])
    return 'is_dynamic":true' in c


def _is_logrotate_enabled(s):
    c = s['runtime_attrs']['content'].get('instance_spec', {}).get('auxDaemons', [])
    return {"type": "LOGROTATE"} in c


def _has_gencfg_volumes(s):
    c = ujson.dumps(s['runtime_attrs']['content'])
    return '"use_volumes":true}' in c


def _has_ssd_resources(s):
    c = ujson.dumps(s['runtime_attrs']['content'])
    return '"storage":"\\/ssd"' in c


def _has_chroot(s):
    instance_spec = s['runtime_attrs']['content']['instance_spec']
    if instance_spec.get('type', 'SANDBOX_LAYERS') != 'SANDBOX_LAYERS':
        # Service definitely has chroot
        return True
    elif instance_spec.get('layersConfig', {}).get('layer'):
        return True
    else:
        return False


def _has_mtn(s):
    i = ujson.dumps(s['runtime_attrs']['content']['instances'])
    return '{"use_mtn":true}' in i


def _has_volumes_enabled(s):
    i = ujson.dumps(s['runtime_attrs']['content']['instances'])
    return '"gencfg_volumes_settings":{"use_volumes":true}' in i


def _get_gencfg_instances(s):
    if s['runtime_attrs']['content']['instances']['chosen_type'] != 'EXTENDED_GENCFG_GROUPS':
        return []
    result = []
    for g in s['runtime_attrs']['content']['instances']['extended_gencfg_groups']['groups']:
        gencfg_url = 'http://api.gencfg.yandex-team.ru/{}/searcherlookup/groups/{}/instances'.format(g['release'],
                                                                                                     g['name'])
        r = requests.get(gencfg_url)
        r.raise_for_status()
        result.extend(r.json()['instances'])
    return result


def _get_yp_pod_ids(s):
    if s['runtime_attrs']['content']['instances']['chosen_type'] != 'YP_POD_IDS':
        return []
    return s['runtime_attrs']['content']['instances'].get('yp_pod_ids', {}).get('pods', [])


def _list_local_paths(s):
    for r in _list_sandbox_resources(s):
        yield r['local_path']
    for r in _list_static_resources(s):
        yield r['local_path']
    for r in _list_template_set_files(s):
        yield r['local_path']
    for r in _list_url_files(s):
        yield r['local_path']


def _get_chosen_type(s):
    return s['runtime_attrs']['content']['instances']['chosen_type']


def _list_gencfg_groups(s):
    if s['runtime_attrs']['content']['instances']['chosen_type'] != 'EXTENDED_GENCFG_GROUPS':
        return []
    return s['runtime_attrs']['content']['instances']['extended_gencfg_groups']['groups']


def _list_sandbox_resources(s):
    return s['runtime_attrs']['content']['resources']['sandbox_files']


def _list_template_set_files(s):
    return s['runtime_attrs']['content']['resources']['template_set_files']


def _list_url_files(s):
    return s['runtime_attrs']['content']['resources']['url_files']


def _list_static_resources(s):
    return s['runtime_attrs']['content']['resources']['static_files']


def _update_service_runtime_attrs(s, sess, comment):
    r = sess.put('{}/v2/services/{}/runtime_attrs/'.format(NANNY_URL, s['_id']), json={
        'snapshot_id': s['runtime_attrs']['_id'],
        'content': s['runtime_attrs']['content'],
        'comment': comment
    })
    r.raise_for_status()


def _update_service_info_attrs(s, sess, comment):
    r = sess.put('{}/v2/services/{}/info_attrs/'.format(NANNY_URL, s['_id']), json={
        'snapshot_id': s['info_attrs']['_id'],
        'content': s['info_attrs']['content'],
        'comment': comment
    })
    r.raise_for_status()


def _update_service_auth_attrs(s, sess, comment):
    r = sess.put('{}/v2/services/{}/auth_attrs/'.format(NANNY_URL, s['_id']), json={
        'snapshot_id': s['auth_attrs']['_id'],
        'content': s['auth_attrs']['content'],
        'comment': comment,
    })
    r.raise_for_status()


def _get_snapshot_current_states(s):
    return s['current_state']['content']['active_snapshots']


def _get_snapshot_target_states(s):
    return s['target_state']['content']['snapshots']


def _get_deploy_engine(s):
    return s['runtime_attrs']['content']['engines']['engine_type']


def _get_users(s):
    rv = set(s['auth_attrs']['content']['owners']['logins'])
    rv.update(s['auth_attrs']['content']['ops_managers']['logins'])
    rv.update(s['auth_attrs']['content']['conf_managers']['logins'])
    return sorted(rv)


def _get_groups(s):
    rv = set(s['auth_attrs']['content']['owners']['groups'])
    rv.update(s['auth_attrs']['content']['ops_managers']['groups'])
    rv.update(s['auth_attrs']['content']['conf_managers']['groups'])
    return sorted(rv)


def _get_binds(s):
    instance_spec = s['runtime_attrs']['content']['instance_spec']
    return instance_spec.get('layersConfig', {}).get('bind', [])


def _has_policy_coredump(s):
    d = ujson.dumps(s['runtime_attrs']['content'].get('instance_spec', {}))
    return 'COREDUMP' in d


def _quotas_enabled(s):
    return s['info_attrs']['content'].get('disk_quotas', {}).get('policy') == 'ENABLED'


def _get_versioned_instancectl_version(s):
    return s['runtime_attrs']['content'].get('instance_spec', {}).get('instancectl', {}).get('version', '-')


def _has_shard(s):
    return s['runtime_attrs']['content']['resources'].get('sandbox_bsc_shard')


def _get_shard_type(s):
    return s['runtime_attrs']['content']['resources'].get('sandbox_bsc_shard', {}).get('chosen_type')


def _get_instance_type(s):
    return s['runtime_attrs']['content'].get('instance_spec', {}).get('type')


def _parse_gencfg_version(v):
    try:
        _, a, b = v.split('-')
        return int(a), int(b[1:])
    except ValueError:
        return 0, 0


def _process_all_services(session, sb_client, zk):
    resp = session.get('{}/api/repo/ListSummaries/'.format(NANNY_URL))
    resp.raise_for_status()
    service_count = resp.json()['total']
    req_limit = 200
    req_offset = 0
    while req_offset < service_count:
        print(req_offset, service_count, file=sys.stderr)
        response = session.get('{}/v2/services'.format(NANNY_URL), params={'skip': req_offset,
                                                                           'limit': req_limit})

        if not response.ok:
            print('Could not get service descriptions')
            response.raise_for_status()

        services = response.json()['result']

        for service in services:
            s_id = service['_id']
            _process_service(s_id, service, session, sb_client, zk)
        req_offset += req_limit


def _process_services_list(session, s_ids, sb_client, zk):
    for s_id in s_ids:
        resp = session.get('{}/v2/services/{}/'.format(NANNY_URL, s_id))
        resp.raise_for_status()
        s = resp.json()
        _process_service(s_id, s, session, sb_client, zk)


def _is_sync_states_enabled(s_id, zk):
    path = '{}/{}'.format(PATH_EMIT, s_id)
    try:
        d, _ = zk.get(path)
    except kazoo.exceptions.NoNodeError:
        return False
    return d == b'1'


def _get_category(s):
    return s['info_attrs']['content']['category']


def _get_replication_policy(session, s_id):
    resp = session.get('{}/api/repo/GetReplicationPolicy/?policyId={}'.format(NANNY_URL, s_id))
    if resp.status_code == 404:
        return None
    return resp.json()['policy']


def _get_replication_policy_type(session, s_id):
    p = _get_replication_policy(session, s_id)
    if not p:
        return None
    return p['spec'].get('replicationMethod', 'REPLACE')


def _list_containers_and_init_containers(s):
    spec = s['runtime_attrs']['content'].get('instance_spec', {})
    for c in spec.get('containers', []):
        yield c
    for c in spec.get('initContainers', []):
        yield c


results = set()


FAIL_START = datetime.datetime(2019, 7, 26, 15, 00, 0)
FAIL_END = datetime.datetime(2019, 7, 16, 10, 0, 0)


HQ_CLIENTS = []


class UserInfo(object):

    __slots__ = ['login', 'is_robot', 'is_dismissed', 'departments']

    def __init__(self, login, is_robot, is_dismissed, departments):
        self.login = login
        self.is_robot = is_robot
        self.is_dismissed = is_dismissed
        self.departments = departments


def _get_hq_clients():
    if HQ_CLIENTS:
        return HQ_CLIENTS
    f_client = federated_stub.FederatedClusterServiceStub(
        requests_client.RetryingRpcClient('http://federated.yandex-team.ru/rpc/federated/',
                                          request_timeout=10)
    )
    r = federated_pb2.FindClustersRequest()
    clusters = f_client.find_clusters(r).value
    for cluster in clusters:
        hq_client = hq_stub.InstanceServiceStub(
            requests_client.RetryingRpcClient(cluster.spec.endpoint.url + 'rpc/instances/', request_timeout=10)
        )
        HQ_CLIENTS.append(hq_client)
    return HQ_CLIENTS


def _get_hq_total_instances_count(s_id):
    find_req = hq_pb2.GetInstanceStatsRequest()
    find_req.filter.service_id = s_id
    total = 0
    for c in _get_hq_clients():
        resp = c.get_instance_stats(find_req)
        total += resp.total_count
    return total


USERS = {}
GROUPS = {}
STATS = {}
LINES = {}
SP_DEPARTMENT_ID = 4112
SP_SERVICE_ID = 851
SP_EXCEPTIONS = (
    'osol',
)


def _is_search_portal_group(session, group_id):
    g = GROUPS.get(group_id)
    if not g:
        g = get_group(session, group_id)
        GROUPS[group_id] = g
    return g


def get_group(session, group_id):
    resp = session.get('https://staff-api.yandex-team.ru/v3/groups?_one=1&id={}&_fields=ancestors,type,department,service,parent'.format(group_id))
    if resp.status_code == 404:
        return None
    try:
        resp.raise_for_status()
    except Exception:
        return None
    j = resp.json()
    if 'type' not in j:
        return None
    if j['type'] == 'department':
        if j['department']['id'] == SP_DEPARTMENT_ID:
            return True
        for g in j['ancestors']:
            if g['department']['id'] == SP_DEPARTMENT_ID:
                return True
    elif j['type'] == 'service':
        if j['service']['id'] == SP_SERVICE_ID:
            return True
        resp = session.get('https://abc-back.yandex-team.ru/api/v3/services/{}/'.format(j['service']['id']))
        try:
            resp.raise_for_status()
        except Exception:
            return None
        for a in resp.json()['ancestors']:
            if a['id'] == SP_SERVICE_ID:
                return True
    elif j['type'] == 'servicerole':
        if j['parent']['service']['id'] == SP_SERVICE_ID:
            return True
        resp = session.get('https://abc-back.yandex-team.ru/api/v3/services/{}/'.format(j['parent']['service']['id']))
        try:
            resp.raise_for_status()
        except Exception:
            return None
        for a in resp.json()['ancestors']:
            if a['id'] == SP_SERVICE_ID:
                return True

    return False


def _get_user_info(session, login):
    r = USERS.get(login)
    if r:
        return r
    resp = session.get('https://staff-api.yandex-team.ru/v3/persons?_pretty=1&_one=1&login={}&_fields=department_group,official'.format(login))
    if resp.status_code == 404:
        return None
    resp.raise_for_status()
    u = resp.json()
    g = u['department_group']
    departments = set()
    departments.add(g['department']['id'])
    departments.add(g['parent']['department']['id'])
    for a in g['ancestors']:
        departments.add(a['department']['id'])
    info = UserInfo(
        is_robot=u['official']['is_robot'],
        is_dismissed=u['official']['is_dismissed'],
        departments=departments,
        login=login,
    )
    USERS[login] = info
    return info


def _is_search_portal_man(u):
    if u.login in SP_EXCEPTIONS:
        return True
    for g in u.departments:
        if g == SP_DEPARTMENT_ID:
            return True
    return False

PS = {}

for cluster in ['man', 'sas', 'vla', 'myt', 'iva']:
    with open('/home/alonger/ya/stats/{}.txt'.format(cluster)) as fd:
        for l in fd:
            _, ps_id, _, cpu, _, ram, _ = l.split()
            ps_id = ps_id.strip('"')
            ps = PS.get(ps_id)
            if not ps:
                ps = {'c': 0, 'm': 0, 'i': 0}
                PS[ps_id] = ps
            ps['c'] += int(cpu.rstrip('u')) / 1000.0
            ps['m'] += int(ram.rstrip('u')) / 1024.0 / 1024.0 / 1024.0
            ps['i'] += 1

sb_client = sandbox.SandboxClient(oauth_token=os.getenv('SANDBOX_OAUTH'))

import time

CACHE = {}

def _get_sandbox_resource_content(s):
    p = (s['task_id'], s['resource_type'])
    g = CACHE.get(p)
    if g:
        return g
    for res in sb_client.list_resources(s['task_id'], s['resource_type']):
        print('GET')
        resp = None
        for i in xrange(10):
            try:
                resp = sb_client._session.get(res['proxy_url'].replace('https', 'http'), timeout=2)
                resp.raise_for_status()
                c = resp.content
                CACHE[p] = c
                return c
            except Exception:
                time.sleep(2)
                print('FAIL...', i)
    return ''


def _process_service(s_id, s, session, sb_client, zk):
    shard = s['runtime_attrs']['content']['resources'].get('sandbox_bsc_shard')
    if not shard:
        return
    if 'SANDBOX_SHARDMAP' in ujson.dumps(shard):
        print(s_id)
    return
    try:
        if _get_summary_state(s) == 'OFFLINE':
            return
    except Exception:
        return
    if _get_instance_type(s) == 'OS_CONTAINER':
        print('OK', s_id, _get_hq_total_instances_count(s_id))
        return
    if '\\/sbin\\/init' in ujson.dumps(s['runtime_attrs']['content']['resources']):
        print('OK', s_id, _get_hq_total_instances_count(s_id))
        return
    for f in _list_sandbox_resources(s):
        if f['local_path'] not in ['instancectl.conf', 'loop.conf']:
            continue
        c = _get_sandbox_resource_content(f)
        if '/sbin/init' in c or '\\/sbin\\/init' in c:
            print('OK', s_id, _get_hq_total_instances_count(s_id))
            return
    return
    if s_id.startswith('sup_mongodb_prod_sh') and s_id.endswith('yp'):
        print(s_id)
        p = _get_replication_policy(session, s_id)
        p['spec']['involuntaryReplicationChoice'] = 'DISABLED'
        resp = session.post('{}/api/repo/UpdateReplicationPolicy/'.format(NANNY_URL),
                            json={
                                'meta': p['meta'],
                                'spec': p['spec']
                            })
        resp.raise_for_status()
    return
    if _get_deploy_engine(s) != 'YP_LITE':
        return
    p = _get_replication_policy(session, s_id)
    if not p:
        return
    p_type = p['spec'].get('replicationMethod', 'REPLACE')
    if p_type != 'REPLACE':
        return
    recipe = p['spec'].get('recipeExecutionParameters', {}).get('recipe')
    if recipe != 'replication.default':
        print(s_id.ljust(80), recipe)
    if not recipe or recipe in ['default', 'common']:
        print(p['spec'])
        p['spec']['recipeExecutionParameters']['recipe'] = 'replication.default'
        resp = session.post('{}/api/repo/UpdateReplicationPolicy/'.format(NANNY_URL),
                            json={
                                'meta': p['meta'],
                                'spec': p['spec']
                            })
    return

    data = PS.get(s_id.replace('_', '-'))
    if not data:
        print('FAIL', s_id)
        return
    users = _get_users(s)
    sp_men = set()
    others = set()
    for login in users:
        info = _get_user_info(session, login)
        if not info:
            continue
        if info.is_robot:
            continue
        if _is_search_portal_man(info):
            sp_men.add(login)
        else:
            others.add(login)
    sp_groups = set()
    other_groups = set()
    for g in _get_groups(s):
        group_info = _is_search_portal_group(session, g)
        if group_info is None:
            continue
        if group_info:
            sp_groups.add(g)
        else:
            other_groups.add(g)
    total_groups = len(sp_groups) + len(other_groups)
    total = len(others) + len(sp_men)
    if not total and not total_groups:
        print('FAIL'.ljust(10), s_id)
        return
    if not others and not other_groups:
        verdict = 'OURS'
    elif len(sp_men) > 3 and not others:
        verdict = 'OURS'
    elif len(sp_men) > len(others) and not other_groups:
        verdict = 'OURS'
    elif not sp_men and not sp_groups:
        verdict = 'ALIEN'
    else:
        verdict = 'MAYBE'
    men_percent = int(len(sp_men) * 100.0 / total) if total else '?'
    group_percent = int(len(sp_groups) * 100.0 / total_groups) if total_groups else '?'
    print(
        s_id.ljust(80),
        '{} {} {}'.format(str(data['i']).rjust(5), str(int(data['c'])).rjust(5), str(int(data['m'])).rjust(5)),
        '{}%'.format(men_percent).rjust(4),
        '({}/{}) men'.format(len(sp_men), total).rjust(10).rjust(9),
        '{}%'.format(group_percent).rjust(4),
        '({}/{}) groups'.format(len(sp_groups), total_groups).rjust(10).rjust(9),
        verdict
    )

    # try:
    #     _get_summary_state(s)
    # except Exception:
    #     print('FAIL', s_id)
    #     return
    # if _get_summary_state(s) == 'OFFLINE' and _get_chosen_type(s) == 'EXTENDED_GENCFG_GROUPS':
    #     if total:
    #         print(s_id.ljust(60), str(total).rjust(6), datetime.datetime.fromtimestamp(s['current_state']['content']['summary']['entered'] / 1000))


def main():
    sess = requests.Session()
    sess.headers['Authorization'] = 'OAuth {}'.format(os.getenv('OAUTH'))
    zk_hosts = os.getenv('ZK_HOSTS')
    zk = None
    if zk_hosts:
        zk = KazooClient(zk_hosts)
        zk.start()
    # _process_services_list(sess, ['yp-test-alonger'], sb_client, zk)
    _process_all_services(sess, sb_client, zk)


if __name__ == '__main__':
    main()
