from __future__ import print_function

import os
import sys
import subprocess
import shutil
import compileall
import re
from argparse import ArgumentParser, Namespace


BETA = ''
_DEBUG_ = False

DEPENDENCIES = 'api kernel library'

parser = ArgumentParser()
parser.add_argument('--beta', '-B', action='store_true', default=False, help='Build beta version of packages')
parser.add_argument('--debug', '-d', action='store_true', default=False)
parser.add_argument('--snapshot', '-s', default='snapshot', help='Path to created snapshot')
parser.add_argument('--deployment')  # ignored, backward compatibility
parser.add_argument('--tag', help='build with kernel and library from specific skynet tag')
options = Namespace()

tag = None

serviceName = 'cqudp'
tgzFile = '{0}.tgz'.format(serviceName)

buildDir = 'build'
distDir = 'dist'

pathSep = ':' if os.name == 'posix' else ';'


class CmdRe(object):
    def __init__(self, pattern, func):
        self.pattern = pattern
        self.re = re.compile(pattern)
        self.func = func

    def __str__(self):
        return self.pattern

    def __repr__(self):
        return '<{0}>'.format(self)

    def __getattr__(self, item):
        return getattr(self.re, item)


class CmdReSet():
    def __init__(self, *args):
        self.elements = args

    def __iter__(self):
        return iter(self.elements)

    def __contains__(self, item):
        return any((re.match(item) for re in self.elements))

    def get(self, item, default=None):
        for cmdRe in self.elements:
            if cmdRe.match(item):
                return cmdRe.func
        return default


def envSafe(func, *args, **kwargs):
    try:
        func(*args, **kwargs)
    except EnvironmentError:
        pass


def shellCall(cmd, setPythonPath=True):
    if setPythonPath:
        env = dict(os.environ)
        path = '{0}{1}{2}'.format(os.path.join(options.snapshot, 'lib'), pathSep, env.get('PYTHONPATH', ''))
        env['PYTHONPATH'] = path
    else:
        env = None

    print('Calling `{0}`'.format(cmd))
    if not _DEBUG_:
        stdout = subprocess.PIPE
        stderr = subprocess.PIPE
    else:
        stdout = None
        stderr = None

    process = subprocess.Popen(cmd, shell=True, stdout=stdout, stderr=stderr, env=env)
    outs = process.communicate()
    if process.returncode:
        print(outs[1])
        raise SystemExit(1)
    return outs[0]


def cleanup(full=False):
    print('cleanup {0}'.format(serviceName))

    if full:
        envSafe(shutil.rmtree, options.snapshot, True)
        envSafe(os.remove, tgzFile)
        envSafe(shutil.rmtree, '_contrib', True)

        for filename in os.listdir('.'):
            if filename.endswith('.egg'):
                envSafe(os.unlink, filename)

    envSafe(shutil.rmtree, buildDir, True)
    envSafe(shutil.rmtree, distDir, True)

    for fileName in os.listdir('.'):
        if fileName.endswith('.egg-info'):
            envSafe(shutil.rmtree, fileName, True)

    for root, files, dirs in os.walk('.'):
        for fileName in files:
            filePath = os.path.join(root, fileName)
            if (fileName.endswith('.pyc') or fileName.endswith('.pyo')) and (os.path.isfile(filePath)):
                envSafe(os.remove, filePath)


def fullCleanup():
    cleanup(True)


def ensureContrib():
    print('looking for skynet dependencies')
    if tag is None:
        base = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/skynet/'
    else:
        base = 'svn+ssh://arcadia.yandex.ru/arc/tags/skynet/%s/skynet/' % (tag,)

    for target in ('kernel', 'library', 'api'):
        dest = os.path.join('_contrib', target)
        destBase = os.path.dirname(dest)
        found = False
        for variant in ('../..', '../../skynet', '../skytrunk-hg'):
            path = os.path.join(variant, target)
            if os.path.exists(path) and os.path.isdir(path) and os.path.exists(os.path.join(path, '__init__.py')):
                found = True
                break
        if found:
            print('{} found in {}'.format(target, os.path.realpath(path)))
            if not os.path.exists(destBase):
                os.makedirs(destBase)
            if os.path.exists(dest):
                os.unlink(dest)
            os.symlink(os.path.abspath(path), dest)
        else:
            link = base + target
            print('{} not found, checking out from {}'.format(target, link))
            shellCall('svn co {} "{}"'.format(link, dest))


def prepare():
    print('preparing directories')
    for path in (
        os.path.join(options.snapshot, 'bin'),
        os.path.join(options.snapshot, 'lib'),
        os.path.join('_contrib'),
    ):
        if not os.path.exists(path):
            os.makedirs(path)
    ensureContrib()


def makeScsd():
    print("Adding cqudp.scsd")
    shutil.copy(os.path.join('.', 'cqudp.scsd'), options.snapshot)


def makeBundle():
    fullCleanup()
    prepare()
    makeScsd()
    envSafe(shutil.rmtree, distDir, True)

    shellCall(
        '"{0}" setup_new.py bdist_wheel {1}'.format(
            sys.executable, BETA
        ),
    )
    shellCall('rm -rf *.egg-info')
    shellCall('"{0}" -m virtualenv --system-site-packages ./{BD}/ve'.format(
        sys.executable,
        BD=buildDir,
    ))
    shellCall('"./{BD}/ve/bin/pip" install -i "http://pypi.yandex-team.ru/simple" pip==1.5.6'.format(BD=buildDir))
    shellCall('"./{BD}/ve/bin/pip" install --no-index --no-deps --compile -f ./{DD} skynet_cqudp_service{BA}'.format(
        BD=buildDir,
        DD=distDir,
        BA='-beta' if options.beta else '',
    ))
    shellCall("cp -r ./{BD}/ve/bin/* ./{SD}/bin/".format(
        BD=buildDir,
        SD=options.snapshot,
    ))
    shellCall("cp -r ./{BD}/ve/lib/python2.7/site-packages/* ./{SD}/lib/".format(
        BD=buildDir,
        SD=options.snapshot,
    ))
    for candidate in ('lib/pip*', 'lib/setuptools*', 'lib/*.py*', 'lib/*.pth', 'lib/_*',
                      'bin/python*', 'bin/easy*', 'bin/pip*', 'bin/activate*'):
        shellCall('rm -rf ./{SD}/{CD}'.format(SD=options.snapshot, CD=candidate))
    compileall.compile_dir(options.snapshot)

    cleanup()


def makeDevelop():
    cleanup(True)
    prepare()
    compileall.compile_dir(options.snapshot)
    makeScsd()

    # for target in ('client', 'server'):
    for target in ['new']:
        shellCall(
            '"{0}" setup_{1}.py develop -d {SD}/lib -s {SD}/bin -Z -O2 -m {BA}'.format(
                sys.executable, target, SD=options.snapshot, BA=BETA
            )
        )


def makePypi():
    cleanup()
    prepare()
    envSafe(shutil.rmtree, distDir, True)

    shellCall(
        '"{0}" setup_pypi.py bdist_wheel --universal -d ./{DD} {1} upload -r yandex'.format(
            sys.executable, BETA,
            DD=distDir,
        ),
    )
    shellCall('rm -rf *.egg-info')

    cleanup()


def makeTgz():
    makeBundle()
    shellCall('tar -C {0} -czvf {1} .'.format(options.snapshot, tgzFile))


if __name__ == '__main__':
    choices = CmdReSet(
        CmdRe('bund.*', makeBundle),
        CmdRe('dev.*', makeDevelop),
        CmdRe('tgz.*', makeTgz),
        CmdRe('dep.*', lambda: print(DEPENDENCIES)),
        CmdRe('clean.*|cln.*', fullCleanup),
        CmdRe('cli.*|pypi', makePypi),
    )

    parser.add_argument('mode', choices=choices)

    options = parser.parse_args()

    if options.beta:
        BETA = ' --beta'
        serviceName = 'cqudp-beta'
        tgzFile = '{0}.tgz'.format(serviceName)

    if options.debug:
        _DEBUG_ = True

    tag = options.tag

    cmd = choices.get(options.mode)
    cmd()
