# -*- coding: utf-8 -*-

import os
import sys
import pwd
import shutil
import tarfile
import argparse
import tempfile
import subprocess
from cStringIO import StringIO

FILENAME = 'cqudp-beta.tgz'


def configure_log():
    from kernel.util import logging
    from kernel.util.log import LogInstaller

    logInstaller = LogInstaller(None, True, True, True)
    logInstaller.install()

    ch = logging.StreamHandler()
    ch.setLevel(logging.DEBUG)

    logging.initialize(level=logging.DEBUG, handler=ch, logger=logging.getLogger('beta'), appendHost=True)


def log(msg):
    from kernel.util.logging import getLogger
    getLogger('beta').info(msg)


def error(msg):
    from kernel.util.logging import getLogger
    getLogger('beta').error(msg)


def make_bundle():
    log("creating package")
    if os.system("{} build.py --beta tgz".format(sys.executable)):
        error("cannot build tgz")
        raise RuntimeError("cannot build tgz!")

    data = open(FILENAME, 'rb').read()
    log("created {} ({} bytes)".format(FILENAME, len(data)))
    return data


class Launcher(object):
    def __init__(self, data, port=None, nowait=False, makelinks=False, uppy=False, user=None):
        self.osUser = user or pwd.getpwuid(os.getuid()).pw_name
        self.data = data
        self.port = port
        self.nowait = nowait
        self.makelinks = makelinks
        self.uppy = uppy

    def __str__(self):
        return 'CQudp beta runner'

    def _find_egg(self, tmpdir):
        for filename in os.listdir(os.path.join(tmpdir, 'lib')):
            if filename.endswith('.egg'):
                return os.path.join(tmpdir, 'lib', filename)

    def __call__(self):
        from api.cqueue.log import RemoteLogHandler
        from kernel.util import logging
        log = logging.getLogger('beta')
        log.setLevel(logging.DEBUG)
        log.addHandler(RemoteLogHandler(implementataion='cqudp'))
        tmpdir = tempfile.mkdtemp()
        log.debug("Created tempdir {}".format(tmpdir))
        try:
            io = StringIO(self.data)
            tgz = tarfile.open(fileobj=io, mode='r:gz')
            tgz.extractall(path=tmpdir)

            if self.makelinks:
                os.symlink(tmpdir, os.path.join(tmpdir, 'service.link'))
                os.rename(os.path.join(tmpdir, 'service.link'), '/Berkanavt/supervisor/services/cqudp-beta')

            if self.uppy:
                if os.system("/skynet/startup/up.py restart cqudp-beta"):
                    raise RuntimeError("Cannot start service!")
            else:

                env = {'PYTHONPATH': ':'.join((self._find_egg(tmpdir), '/Berkanavt/supervisor/skynet'))}
                proc = subprocess.Popen(['/skynet/python/bin/python',
                                         os.path.join(tmpdir, 'bin', 'cqudp'),
                                         'server',
                                         '--port', str(self.port),
                                         '-t', tmpdir,
                                         ],
                                        preexec_fn=os.setpgrp if self.nowait else lambda: None,
                                        env=env)
                if not self.nowait:
                    proc.communicate()
                    if proc.returncode:
                        raise RuntimeError("Server exited with {}".format(proc.returncode))
        finally:
            if not self.uppy and not self.makelinks and not self.nowait:
                shutil.rmtree(tmpdir)


def resolve_hosts(hosts):
    from library.sky.hostresolver import Resolver

    hosts = ' '.join(hosts)
    log('resolving hosts: {}'.format(hosts))
    hosts = list(Resolver().resolveHosts(hosts))
    log('resolved {} hosts: {}'.format(len(hosts), hosts))

    return hosts


def port():
    from library.config import query
    from kernel.util import beta

    beta.set()
    basePort = query('skynet.services.cqudp', 'config')['server']['bus_port']

    from api.config import basePortOffset
    offset = basePortOffset()

    return basePort + offset + beta.level()


def upload(client, data, hosts, args):
    p = (not args.uppy and args.port) or port()
    log('servers will start on port {}'.format(p))
    log('starting cq session')
    task = Launcher(data, port=p, nowait=args.nowait, makelinks=args.links, uppy=args.uppy, user=args.user)

    with client:
        with client.run(hosts, task) as s:
            for h, r, e in s.wait():
                if e is None:
                    log('{} ended with {}'.format(h, r))
                else:
                    error('{} failed with {}'.format(h, e))


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('-l', '--links', action='store_true', default=False, help='Make symlinks in Berkanavt')
    parser.add_argument('-s', '--uppy', action='store_true', default=False, help='Start service with up.py (implies -l)')
    parser.add_argument('-n', '--nowait', action='store_true', default=False, help='Start in background. Ignored if -s is set')
    parser.add_argument('-u', '--user', default=None, help='User to be used')
    parser.add_argument('-p', '--port', default=None, help='Custom port to run on (ignored if uppy isset)')
    parser.add_argument('HOSTS', nargs=argparse.REMAINDER, help='Hosts to run on')

    args = parser.parse_args()

    configure_log()

    from api.cqueue import Client
    client = Client('cqudp')

    data = make_bundle()
    hosts = resolve_hosts(args.HOSTS)
    upload(client, data, hosts, args)


if __name__ == '__main__':
    raise SystemExit(main())
