from __future__ import print_function

import sys
import six
import os
import shutil
import argparse
from collections import defaultdict

from setuptools.command.bdist_egg import walk_egg, log
from setuptools.command.build_py import build_py
from setuptools.command.develop import develop
from setuptools.command.install import install

try:
    from setuptools.command.bdist_egg import NS_PKG_STUB
except ImportError:
    NS_PKG_STUB = '__import__("pkg_resources").declare_namespace(__name__)'

try:
    import setuptools
    import pkg_resources
except ImportError:
    print('Please install distribute from PyPi first.', file=sys.stderr)
    exit(1)


path = os.path.join(os.path.dirname(os.path.abspath(__file__)), '_contrib')
sys.path.insert(1, path)

parser = argparse.ArgumentParser()
parser.add_argument('--beta', '-B', action='store_true', help='run as beta', default=False)
options, new_argv = parser.parse_known_args()
sys.argv = [sys.argv[0]] + new_argv

# print(list(pkg_resources.iter_entry_points('egg_info.writers')))

_skynetPrefix = 'ya.skynet'
_servicePrefix = '{0}.services.cqudp'.format(_skynetPrefix)

_packageDirs = {
    _servicePrefix: 'src',
    '{0}.util'.format(_skynetPrefix): '_contrib/kernel/util',
    '{0}.library'.format(_skynetPrefix): '_contrib/library',
}

_packagePaths = {}

for packageName, packageDir in _packageDirs.items():
    packageParts = packageName.split('.')
    lastPart = packageParts.pop()
    dictNode = _packagePaths
    for packagePart in packageParts:
        dictNode = dictNode.setdefault(packagePart, {})
    dictNode[lastPart] = packageDir


def packagePath(packageName):
    dictNode = _packagePaths

    packageParts = packageName.split('.')
    packageParts.reverse()

    while packageParts:
        dictNode = dictNode.get(packageParts[-1])
        packageParts.pop()
        if dictNode is None or isinstance(dictNode, six.string_types):
            break

    if isinstance(dictNode, six.string_types):
        packageParts.reverse()
        return os.path.abspath(os.path.join(dictNode, *packageParts)), len(packageParts)
    else:
        packageParts = packageName.split('.')
        return os.path.abspath(os.path.join(*packageParts)), len(packageParts)


def make_ns_init(nsDir):
    fileName = os.path.join(nsDir, '__init__.py')
    if not os.path.isfile(fileName):
        with open(fileName, 'w') as f:
            f.write(NS_PKG_STUB)


def makePackageDirs(path, packageParts):
    for packagePart in packageParts:
        path = os.path.join(path, packagePart)
        if not os.path.isdir(path):
            os.makedirs(path)


class mbuild_py(build_py):
    def make_init_files(self):
        """Create missing package __init__ files"""
        init_files = []
        for base, dirs, files in walk_egg(self.build_lib):
            if base == self.build_lib:
                # don't put an __init__ in the root
                continue

            if '__init__.py' not in files:
                pkg = base[len(self.build_lib) + 1:].replace(os.sep, '.')
                if self.distribution.has_contents_for(pkg):
                    log.warn("Creating missing __init__.py for %s", pkg)
                    filename = os.path.join(base, '__init__.py')
                    if not self.dry_run:
                        make_ns_init(base)
                    init_files.append(filename)

        return init_files

    def build_packages(self):
        build_py.build_packages(self)
        self.make_init_files()


class mdevelop(develop):
    # def enshurePackage
    # TODO: Dont recreate full structure

    def install_for_development(self):
        self.reinitialize_command('egg_info')
        self.run_command('egg_info')
        # Build extensions in-place
        self.reinitialize_command('build_ext', inplace=1)
        self.run_command('build_ext')
        self.install_site_py()  # ensure that target dir is site-safe
        if setuptools.bootstrap_install_from:
            self.easy_install(setuptools.bootstrap_install_from)
            setuptools.bootstrap_install_from = None

        ei = self.get_finalized_command("egg_info")

        build_py = self.get_finalized_command('build_py')

        # create an .egg-link in the installation dir, pointing to our egg
        log.info('Creating structure')

        if not self.dry_run:
            log.info('Linking sources')

            packages = list(self.distribution.packages)
            packages.sort()

            packages.reverse()

            for package in packages:
                srcPath, packageDepths = packagePath(package)
                packageDepths += 1
                assert os.path.exists(srcPath)
                packageParts = package.split('.')
                dstPath = os.path.join(self.install_dir, *packageParts)

                if os.path.exists(dstPath):
                    continue

                makePackageDirs(self.install_dir, packageParts)

                for _ in six.moves.xrange(len(packageParts)):
                    if packageDepths > 0:
                        packageDepths -= 1
                        for fileName in os.listdir(srcPath):
                            if (
                                os.path.isfile(os.path.join(srcPath, fileName)) and
                                not os.path.exists(os.path.join(dstPath, fileName)) and
                                not fileName.endswith('.pyc') and
                                not fileName.endswith('.pyo')
                            ):
                                os.symlink(os.path.join(srcPath, fileName), os.path.join(dstPath, fileName))
                        srcPath = os.path.dirname(srcPath)

                    make_ns_init(dstPath)
                    dstPath = os.path.dirname(dstPath)

        dataFiles = defaultdict(set)

        for package, _, _, fileNames in build_py.data_files:
            dataFiles[package].update(fileNames)

        for package in packages:
            srcPath, _ = packagePath(package)
            assert os.path.exists(srcPath)
            packageParts = package.split('.')
            dstPath = os.path.join(self.install_dir, *packageParts)

            for fileName in dataFiles[package]:
                target = os.path.join(dstPath, fileName)
                targetDir = os.path.dirname(target)
                if not os.path.exists(targetDir):
                    self.mkpath(os.path.dirname(target))
                if not os.path.exists(target):
                    os.symlink(os.path.join(srcPath, fileName), target)

        # postprocess the installed distro, fixing up .pth, installing scripts,
        # and handling requirements
        self.process_distribution(None, self.dist, not self.no_deps)

        if not self.dry_run:
            log.info('Moving EGG-INFO')
            # shutil.move(os.path.join(self.egg_path, ei.egg_info), os.path.join(self.eggDir, 'EGG-INFO'))
            shutil.move(os.path.join(self.egg_path, ei.egg_info), self.install_dir)


class minstall(install):
    def run(self):
        install.run(self)

        for path, dirs, files in os.walk(self.install_purelib):
            if path == self.install_purelib:
                continue

            if '__init__.py' not in files:
                pkg = path[len(self.install_purelib):].replace(os.sep, '.')
                if self.distribution.has_contents_for(pkg):
                    log.warn('Creating missing __init__.py for %s', pkg)
                    if not self.dry_run:
                        make_ns_init(path)


setupArgs = dict(
    version='0.2.4',
    zip_safe=False,
    install_requires=[
        'pycrypto>=2.5',
        'PyYAML',
        'six',
        'msgpack_python>=0.2',
        'gevent',
    ] + (['pywin32'] if os.name != 'posix' else []),
    package_dir=_packageDirs,
    namespace_packages=[_servicePrefix.split('.')[0]],
    cmdclass={
        'build_py': mbuild_py,
        'develop': mdevelop,
        'install': minstall,
    },
    url='https://wiki.yandex-team.ru/Skynet/',
)


def packageOnly(skyPackage):
    yield '{0}.{1}'.format(_skynetPrefix, skyPackage)


def packageRecursive(skyPackage):
    package = '{0}.{1}'.format(_skynetPrefix, skyPackage)
    yield package
    for subPackage in setuptools.find_packages(os.path.join(packagePath(package)[0])):
        yield '{0}.{1}'.format(package, subPackage)
