from __future__ import absolute_import

import time

import six

from ..utils import singleton, monotime
from ..exceptions import CQueueAuthenticationError, CQueueAuthorizationError, CQueueExecutionError, CQueueRuntimeError


class Metric(object):
    def __init__(self, host_num):
        self.host_num = host_num

    def handle(self, err, res):
        raise NotImplementedError()

    def report(self):
        raise NotImplementedError()


class ErrorMetric(Metric):
    class ErrorResult(object):
        def __init__(self, error):
            if isinstance(error, CQueueAuthenticationError) or isinstance(error, CQueueAuthorizationError):
                self.type = 'AccessError'
            elif isinstance(error, CQueueRuntimeError):
                self.type = 'RuntimeError'
            elif isinstance(error, CQueueExecutionError):
                self.type = 'UserError'
            else:
                self.type = 'Unknown'

            self.error = str(error)
            self.count = 1

    def __init__(self, host_num):
        super(ErrorMetric, self).__init__(host_num)

        self.errors = dict()
        self.err_results = 0

    def handle(self, err, res):
        if res is None and err is not None and not isinstance(err, StopIteration):
            # error occurred on the server side
            # do we have same report already?
            rep = self.errors.get(str(err), None)

            if rep is not None:
                rep.count += 1
            else:
                self.errors[str(err)] = self.ErrorResult(err)

            self.err_results += 1
            return 1

        # data is not handled
        return 0

    def report(self):
        errors_by_type = dict()
        for error in list(self.errors.values()):
            # merge errors by type
            value = errors_by_type.get(error.type)
            if value is None:
                errors_by_type[error.type] = (error.count, '{}: {}'.format(error.count, error.error))
            else:
                count = value[0] + error.count
                desc = value[1] + ',{}: {}'.format(error.count, error.error)
                errors_by_type[error.type] = (count, desc)

        for key, value in six.iteritems(errors_by_type):
            report().info('{} {} {} {} {}'.format(key, time.time(), self.host_num, value[0], value[1]))


class OverheadMetric(Metric):
    def __init__(self, host_num):
        super(OverheadMetric, self).__init__(host_num)

        self.start_time = monotime()
        self.total_user_time = 0.0
        self.max_user_time = 0.0
        self.user_results = 0

    def handle(self, err, res):
        # we expect list as a result: <data> <metric>
        if res is not None and len(res) == 2:
            self.max_user_time = max(self.max_user_time, res[1])
            self.total_user_time += res[1]
        # dirty hack: we need info that task is complete, for now we use StopIteration exception
        elif isinstance(err, StopIteration):
            # task has been completed
            self.user_results += 1
            return 1

        return 0

    def report(self):
        if self.user_results > 0:
            # report overhead if we have at least one user result
            total_time = monotime() - self.start_time
            report().info('Overhead {} total_hosts: {} total_time: {:f} total_user_time: {:f} number_of_results: {} '
                          'delta: {:f}'
                          .format(time.time(), self.host_num, total_time, self.total_user_time, self.user_results,
                                  total_time - self.max_user_time))


class CqueueMetrics(object):
    def __init__(self, host_num):
        self.host_num = host_num
        self.metrics = [OverheadMetric(host_num), ErrorMetric(host_num)]

    def update(self, err, res):
        for metric in self.metrics:
            self.host_num -= metric.handle(err, res)

        self._report()

    def _report(self):
        if self.host_num > 0:
            # we wait for results from all hosts before writing the metric report
            # if session is not complete we ignore this result
            return

        for metric in self.metrics:
            metric.report()


@singleton
def report():
    from api.logger import constructLogger
    return constructLogger(source='cqudp.metrics', app='cqudp', filename='cqudp-metrics.log')
