import six

from ..window import MultiWindow, IndexedWindow
from .. import cfg


class Scheduler(object):
    SILENT = 0
    TASK_STARTING = 1
    TASK_STARTED = 2
    REPLIES_STARTED = 3
    TASK_FINISHED = 4

    def __init__(self, hosts, t):
        self.status = {h: HostStatus() for h in hosts}
        self.update_time = t
        # TODO (torkve) take value from config, but remember that bus has client/server dualism
        self.retry_period = 15  # Should be equal to messagebus.TOTAL_TIMEOUT. Don't want to make a dependency.
        self.timeout_factor = 1
        self.task_parts = 1  # if >1, then part_requested must be called

    def task_sent(self, host, t):
        self.status[host].task_sent = max(t, self.status[host].task_sent)

        self.status[host].outgoing_time = max(t, self.status[host].outgoing_time)

    def heartbeat_sent(self, host, n, t):
        self.status[host].outgoing_time = max(t, self.status[host].outgoing_time)

    def heartbeat_received(self, host, t):
        self.touch(host, t)
        hs = self.status.get(host)
        if hs:
            hs.last_status = max(hs.last_status, Scheduler.TASK_STARTED)

    def result_received(self, host, index, res, t):
        return [(indx, data) for _, _, indx, data in self.response_received(host, index, 'result', res, t)]

    def response_received(self, host, index, label, response, t):
        self.touch(host, t)

        hs = self.status.get(host)
        if hs is None:
            return []

        hs.last_status = max(hs.last_status, Scheduler.REPLIES_STARTED)
        hs.window.put(index, label, response)
        return hs.window.pop()

    def task_done(self, host, t):
        self.touch(host, t)
        self.status[host].done = True
        self.status[host].last_status = Scheduler.TASK_FINISHED

    def part_requested(self, host, sn, t):
        self.touch(host, t)
        hs = self.status[host]
        if hs.next_part < sn:
            hs.next_part_requested = True
        hs.next_part = max(sn, hs.next_part)
        hs.last_status = max(hs.last_status, Scheduler.TASK_STARTING)

    def host_status(self, host):
        hs = self.status.get(host)
        if hs is not None:
            return hs.last_status

    def touch(self, host, t):
        self._update(t)
        hs = self.status.get(host)
        if hs:
            hs.update(t)

    @property
    def heartbeat_period(self):
        return max(cfg.client.Heartbeats.MinimalPeriod, len(self.status) // cfg.client.Heartbeats.HostsDenominator)

    def schedule_heartbeats(self, t, append_rpcid=False):
        tree_data = {}
        for k, v in six.iteritems(self.status):
            if (
                v.next != HostStatus.HB
                or v.done
                or t - v.outgoing_time <= self.heartbeat_period
            ):
                continue

            tree_data[k] = v.window.window.index if not append_rpcid else (v.window.window.index, v.outgoing_messages.idx)

        return tree_data

    def schedule_tasks(self, t):
        task_period = max(self.heartbeat_period, self.retry_period)
        hosts = []
        for k, v in six.iteritems(self.status):
            if (
                v.next == HostStatus.TASK
                and ((t - v.task_sent > task_period) or v.next_part_requested)
            ):
                hosts.append((k, v.next_part))
                v.next_part_requested = False

        return hosts

    @property
    def host_timeout(self):
        return self.retry_period * self.timeout_factor

    def schedule_dead(self, t):
        if t - self.update_time <= self.host_timeout:
            return []

        dead = [
            k
            for k, v in six.iteritems(self.status)
            if not v.done and t - v.incoming_time > self.retry_period
        ]
        for h in dead:
            self.status[h].done = True
        return dead

    def host_dead(self, h):
        self.status[h].done = True

    # TODO: respect t.
    def get_unfinished_hosts(self, t):
        return [k for k, v in six.iteritems(self.status) if not v.done]

    def _update(self, t):
        self.update_time = max(t, self.update_time)

    def rpc_sent(self, host, msg, data):
        h = self.status.get(host)
        if h is not None:
            return h.outgoing_messages.enqueue((msg, data))

    def rpc_requested(self, host, rpcid):
        h = self.status.get(host)
        if h is not None:
            return h.outgoing_messages.seek(rpcid)  # (idx, (msg, data))


class HostStatus(object):
    TASK = 1
    HB = 2

    __slots__ = [
        'task_sent',
        'done',
        'incoming_time',
        'outgoing_time',
        'window',
        'last_status',
        'next_part',
        'next_part_requested',
        'outgoing_messages',
    ]

    def __init__(self):
        self.task_sent = -1
        self.done = False
        self.incoming_time = -1
        self.outgoing_time = -1
        self.window = MultiWindow()
        self.last_status = Scheduler.SILENT
        self.next_part = 0
        self.next_part_requested = False
        self.outgoing_messages = IndexedWindow(None)

    @property
    def next(self):
        if self.done:
            return None
        if not self.running or self.silent or self.next_part_requested:
            return HostStatus.TASK
        return HostStatus.HB

    @property
    def silent(self):
        return self.incoming_time < 0

    @property
    def running(self):
        return self.task_sent >= 0

    def update(self, t):
        self.incoming_time = max(t, self.incoming_time)
