# -*- coding: utf-8 -*-
from __future__ import absolute_import

from collections import defaultdict
import os
import time
import math

import six

from ya.skynet.util.pickle import dumps, Pickler, Unpickler, UnpicklingError
from ya.skynet.util import logging

from ..transport.protocol import Dispatcher, handle
from ..exceptions import CQueueRuntimeError
from ..utils import FdHolder, short, genuuid, debug_log, sleep, bytesio, monotime
from ..eggs import serialize
from ..rpc import RPCDispatcher, RPCObject
from .task import task, task_hash, task_options
from .handle import RHandle
from .unpickling import import_if_permitted
from .metrics import report
from .schedule import Scheduler
from ..poll import Selectable, ResultsQueue
from ..window import IncomingWindow
from .. import cfg, msgpackutils as msgpack


class Session(Dispatcher, Selectable):
    _event_fd = FdHolder('event_fd')
    _event_notify_fd = FdHolder('event_notify_fd')

    def __init__(self,
                 client,
                 signer,
                 hosts,
                 runnable,
                 task_hosts_data=None,
                 task_type='task',
                 task_opts=None,
                 log=None,
                 ):
        super(Session, self).__init__(log=log)

        self._task_opts = task_opts or {}

        self.client = client
        self.results = ResultsQueue(self.client.select_function)
        self.rpc_dispatcher = RPCDispatcher()
        self._event_fd, self._event_notify_fd = None, None

        default_port = client.default_port()
        self.hosts, self.user_hosts, self._task_hosts_data = self.prepare_host_data(hosts,
                                                                                    default_port,
                                                                                    task_hosts_data)

        self.scheduler = Scheduler(list(self.hosts.keys()), monotime())

        self.allow_unsafe_unpickle = self._task_opts.pop('allow_unsafe_unpickle_i_am_sure',
                                                         cfg.client.AllowUnsafeUnpickle)
        retry_timeout = self._task_opts.pop('retry_timeout', cfg.client.RetryTimeout)
        if retry_timeout:
            self.scheduler.retry_timeout = retry_timeout

        self.scheduler.timeout_factor = max(1, int(self._task_opts.pop('timeout_factor', 1)))

        if self._task_opts.get('port_range') and not isinstance(self._task_opts['port_range'], (list, tuple)):
            self._task_opts['port_range'] = cfg.client.Transport.SpecialPortRange

        self.multipart_size = 0
        multipart = self._task_opts.pop('multipart', cfg.client.Multipart)
        if multipart:
            bandwidth = int(self._task_opts.pop('multipart_size', cfg.client.MultipartBandwidth))
            self.multipart_size = int(max(1e5, bandwidth / (len(hosts) ** 0.5)))
        self._create_task(task_type, runnable, signer, self._task_opts)
        if self.multipart_size > 0:
            msg = msgpack.dumps(self._task_msg)
            total = int(math.ceil(float(len(msg)) / self.multipart_size))
            self._task_parts = [
                multipart_msg(msg[x * self.multipart_size:x * self.multipart_size + self.multipart_size],
                              sn=x,
                              total=total,
                              taskid=self.taskid)
                for x in six.moves.xrange(total)
            ]
            self.scheduler.task_parts = total
        else:
            self._task_parts = None

        # reset scheduler update time after task packing (which can be time-consuming)
        self.scheduler._update(monotime())

        debug_log().debug("[%s] serialized task length %d bytes",
                          short(self.taskid),
                          len(self._task_msg['data']))
        debug_log().debug("[%s] signs length %d bytes",
                          short(self.taskid),
                          sum((len(s) + len(fp)) for fp, s in self._task_msg['signs']))

        self._task_msg['options'].update(self._task_opts)
        # FIXME (torkve) what the hell 'options' we're setting here?
        # nevertheless, select_function is not needed and cannot be msgpacked
        self._task_msg['options'].pop('select_function', None)

        self.start_time = monotime()

        self.__log_handler = LogHandler()
        self.rpc_dispatcher.register_type_handler('log', self.__log_handler)

        if self._task_msg['options']['forward_agent']:
            self.__ssh_handler = SshHandler(self.taskid, signer, self.route_rpc, self.log)
            self.rpc_dispatcher.register_type_handler('sshagent', self.__ssh_handler)

        self.session_timeout = self._task_msg['options']['session_timeout']
        self.rpc_confirmations = self._task_opts.get('rpc_confirmations', cfg.client.ConfirmRpcDelivery)

        self._unpickle_metric = defaultdict(int)

    def _find_global(self, module_name, attr_name):
        self._unpickle_metric['%s//%s' % (module_name, attr_name)] += 1
        return import_if_permitted(module_name,
                                   attr_name,
                                   self.allow_unsafe_unpickle,
                                   additional_whitelist=self.client.allowed_unpickles)

    def report_type_stats(self):
        if self._unpickle_metric:
            report().info('[%s] ResTypes %s'
                          % (short(self.taskid),
                             " ".join("%s:%d" % (key, value)
                                      for key, value in six.iteritems(self._unpickle_metric)))
                          )

    def loads(self, data):
        unpickler = Unpickler(bytesio(data))
        unpickler.find_global = self._find_global
        return unpickler.load()

    @classmethod
    def prepare_host_data(cls, hosts, default_port, task_hosts_data):
        if isinstance(hosts, dict):
            # if hosts are already enumerated
            hosts_ = {k: HostEntry(v, default_port) for k, v in list(hosts.items())}
        else:
            hosts_ = dict(enumerate(HostEntry(h, default_port) for h in hosts))

        user_hosts = dict((h.user_host, hostid) for hostid, h in six.iteritems(hosts_))
        task_hosts_data_ = dict((i, dumps(p)) for i, p in enumerate(task_hosts_data)) if task_hosts_data else None
        return hosts_, user_hosts, task_hosts_data_

    def _get_acc_user(self):
        return None

    def get_host_by_id(self, hostid):
        return self.hosts[hostid].user_host

    def _create_task(self, task_type, runnable, signer, task_opts):
        task_opts = task_opts or {}
        # TODO: sign options.
        options = task_options(runnable)

        # TODO: encapsulate params assignment.
        params = {
            'wait_min': 1,
            'wait_max': self.scheduler.heartbeat_period,
            'orphan_timeout': self._orphan_timeout(),
            'aggregate': task_opts.get('aggregate', cfg.client.Transport.Aggregate),
            'pipeline': task_opts.get('pipeline', cfg.client.Transport.Pipeline),
            'msgpack': task_opts.get('msgpack', cfg.client.Transport.Msgpack),
            'netlibus': task_opts.get('netlibus', cfg.client.Transport.Netlibus),
            'loglevel': task_opts.get('loglevel', cfg.client.LogLevel),
        }

        uuid = task_opts.pop('uuid', None)
        dumped = task_opts.pop('objdumped', False)
        self._task_msg = t = task(None, options=options, task_type=task_type, uuid=uuid, acc_user=self._get_acc_user())

        self.log = logging.MessageAdapter(
            self.log,
            fmt="[%(uuid)s] %(message)s",
            data={'uuid': short(self.taskid)},
        )

        obj = self._serialize(runnable,
                              modules=getattr(runnable, 'marshaledModules', None),
                              dumped=dumped)
        t['data'] = obj
        t['params'] = params

        if signer is not None:
            t['signs'].extend(signer.sign(task_hash(t)))

    def _serialize(self, runnable, modules=None, dumped=False):
        if not runnable:
            return b''

        if dumped:
            return runnable

        def set_object_session(obj):
            if getattr(obj, 'needs_session', None) is True:
                obj._set_session(self)
            return None

        io = bytesio()
        pickler = Pickler(io)
        pickler.persistent_id = set_object_session
        pickler.dump(runnable)
        return serialize(io.getvalue(),
                         modules=modules,
                         arcadia_binary=bool(self._task_opts.pop('arcadia_serialize', False)),
                         log=self.log.getChild('eggs'),
                         )

    def make_results_handle(self):
        return RHandle(self, self.results)

    def rpc_send(self, rpctype, rpcid, data, user_hosts, tree_data=None):
        hosts = list(self._find_actual_addrs(user_hosts))
        msg = rpc_msg(self.taskid, rpctype, rpcid, data)
        self.route_rpc(msg, hosts, tree_data)

    def route_rpc(self, msg, hosts, tree_data=None):
        # FIXME enable always after 15.1 is everywhere
        if self.rpc_confirmations:
            tree_data = tree_data or {}
            for hostid, addr in hosts:
                tree_data[hostid] = (self.scheduler.rpc_sent(hostid, msg, tree_data.get(hostid)),
                                     tree_data.get(hostid))
        self.client.route(msg, hosts, data=tree_data)

    def _stop(self):
        for _ in range(cfg.client.StopMessage.Attempts):
            self.client.route(stop_msg(self.taskid),
                              [(hostid, h.actual_addr)
                               for hostid, h in list(self.hosts.items())])
            sleep(cfg.client.StopMessage.Interval)

    def stop(self):
        self.client.spawn(self._stop)

    def handle_result(self, hostid, idx, data):
        try:
            res, err = self.loads(data)
        except UnpicklingError as e:
            res, err = None, CQueueRuntimeError("Host sent result that couldn't be unpickled: %s" % (e,))

        if err:
            self.scheduler.task_done(hostid, monotime())

        self.results.append((self.hosts[hostid].user_host, idx, res, err))
        self._notify()

    def get_good_hosts(self):
        return [(k, h.original_addr) for k, h in six.iteritems(self.hosts)]

    def schedule(self):
        """
        Returns tuple with the following contents:
            0: data for heartbeats to send in the form: {hostid => next_result_index, …}
            1: heartbeat message dict (contains only taskid and orphan_timeout)
            2: tasks to send in the form: [(dests, msg, data), …] where fields are:
                0: destinations of tasks to send in the form: [(hostid, addr), …]
                1: task message dict
                2: data for task to send in the form: {hostid => object, …}
            3: list of hosts to send stop message
            4: stop message
        Note: while in base Session class field 4 always has structure {hostid => str, …},
        currently subclasses may return some other value type, e.g. {hostid => (str, str), …},
        which can only be sent as-is.
        """
        t = monotime()

        if self._check_session_timeout(t):
            return {}, (), [], [], None

        # delayed pop
        to_pop = []
        for h, v in six.iteritems(self.scheduler.status):
            if v.done:
                to_pop.append(h)

        stop_hosts = [(hostid, self.hosts[hostid].actual_addr) for hostid in to_pop]
        stop_message = stop_msg(self.taskid) if stop_hosts else None

        for h in to_pop:
            self.scheduler.status.pop(h, None)

        heartbeats = self.scheduler.schedule_heartbeats(t, self.rpc_confirmations)
        for h, v in six.iteritems(heartbeats):
            self.scheduler.heartbeat_sent(h, v, t)

        tasks = self.scheduler.schedule_tasks(t)
        for h, sn in tasks:
            self.scheduler.task_sent(h, t)

        dead = set(self.scheduler.schedule_dead(t))

        possibly_dead = {h for h in dead
                         if t - self.client.alive_hosts.get(self.hosts[h].hostname, 0) <= self.scheduler.host_timeout}

        dead = dead - possibly_dead

        self.log.debug('scheduled %d dead hosts, %d possibly dead', len(dead), len(possibly_dead))

        self._handle_dead(dead)

        hb_msg = heartbeat_msg(self.taskid, orphan_timeout=self._orphan_timeout())
        msgs = list(self._resolve(tasks))
        return heartbeats, hb_msg, msgs, stop_hosts, stop_message

    def _resolve(self, hosts):
        taskmap = defaultdict(set)
        for hostid, sn in hosts:
            taskmap[sn].add(hostid)
        for sn, hostids in six.iteritems(taskmap):
            hosts = [(hostid, self.hosts[hostid].actual_addr) for hostid in hostids]
            msg = self._task_msg if self.scheduler.task_parts == 1 else self._task_parts[sn]
            data = (
                dict(kv for kv in six.iteritems(self._task_hosts_data) if kv[0] in hostids)
                if sn == 0 and self._task_hosts_data is not None
                else None
            )
            yield hosts, msg, data

    @property
    def taskid(self):
        return getattr(self, '_task_msg', {}).get('uuid')

    def is_empty(self):
        return len(self.scheduler.status) == 0

    def _find_actual_addrs(self, user_hosts):
        for host in user_hosts:
            if isinstance(host, list):
                host = tuple(host)
            if host not in self.user_hosts:
                raise CQueueRuntimeError("Host not known in the session {}: {}".format(short(self.taskid), host))

            hostid = self.user_hosts[host]
            yield hostid, self.hosts[hostid].actual_addr

    def _handle_dead(self, dead):
        for h in dead:
            entry = self.scheduler.host_status(h)
            self.scheduler.status.pop(h, None)

            status = "Host is silent"
            if entry == Scheduler.TASK_STARTING:
                status = "Task was being transferred, but connection has been lost"
            elif entry == Scheduler.TASK_STARTED:
                status = "Task started, but no replies received"
            elif entry == Scheduler.REPLIES_STARTED:
                status = "Host went silent after some replies"
            elif entry == Scheduler.TASK_FINISHED:
                status = "Unexpected: all results were already received"

            self.results.append((self.hosts[h].user_host, 0, None, Timeout(status)))
            self._notify()

    def _check_session_timeout(self, t):
        if self.session_timeout is not None and t - self.start_time > self.session_timeout:
            for k in list(self.scheduler.status.keys()):
                v = self.scheduler.status.pop(k, None)
                if not v.done:
                    self.results.append((self.hosts[k].user_host, v.window.window.index, None, Timeout()))
            return True
        return False

    def _get_event_fd(self):
        if self._event_fd is None:
            self._event_fd, self._event_notify_fd = os.pipe()

        return self._event_fd

    def _notify(self):
        if self._event_notify_fd is not None:
            os.write(self._event_notify_fd, b'1')

    def _is_data_ready(self):
        return bool(self.results)

    def __update_port(self, initiator):
        hostid = initiator[0]
        (host, port) = initiator[1]
        if hostid in self.hosts and port != self.hosts[hostid].updated_port:
            self.hosts[hostid].updated_port = port

    @handle('heartbeat')
    def __heartbeat(self, msg, data, path, hostid, iface):
        path = fix_path(msg, path)
        self.__update_port(path[0])

        hostid = path[0][0]
        if hostid not in self.scheduler.status:
            self.log.debug('ignoring heartbeat %s from %s', short(msg['uuid']), path[0])
            return

        self.log.debug('heartbeat from %s', path[0])
        self.scheduler.heartbeat_received(hostid, monotime())

        requested_rpcid = msg.get('request_rpc')
        if self.rpc_confirmations and requested_rpcid is not None:
            requested_msg = self.scheduler.rpc_requested(hostid, requested_rpcid)
            if requested_msg is not None:
                rpcid, (msg, data) = requested_msg
                self.client.route(msg, [path[0]], {hostid: (rpcid, data)})

    @handle('result')
    def __result(self, msg, data, path, hostid, iface):
        path = fix_path(msg, path)
        self.__update_port(path[0])

        idx = msg['index']
        self.log.debug('result #%s from %s', idx, path[0])

        hostid, host = path[0]
        self._check_result(path[0], msg)

    def _check_result(self, initiator, msg):
        hostid, host = initiator
        idx = msg['index']

        if hostid not in self.scheduler.status:
            self.log.debug('ignored result from %s', host)
            return

        for idx, data in self.scheduler.result_received(hostid, idx, msg['result'], monotime()):
            self.handle_result(hostid, idx, data)

    @handle('rpc')
    def __rpc(self, msg, data, path, hostid, iface):
        path = fix_path(msg, path)
        self.__update_port(path[0])

        self.rpc_dispatcher.dispatch(path[0], msg)

    @handle('response')
    def __response(self, msg, data, path, hostid, iface):
        path = fix_path(msg, path)
        self.__update_port(path[0])

        hostid, host = path[0][:2]

        if hostid not in self.scheduler.status:
            self.log.debug('ignored response from %s', host)
            return

        content = msg['content']
        index = msg['index']
        inner_type = content['type']
        inner_msg = content['data']

        self.log.debug('response/%s #%s from %s', inner_type, index, path[0])

        for _, label, index, data in self.scheduler.response_received(hostid,
                                                                      index,
                                                                      inner_type,
                                                                      inner_msg,
                                                                      monotime()):
            self.handle_response(hostid, path[0], label, index, data)

    def handle_response(self, hostid, addr, label, index, data):
        try:
            data = self.loads(data)
        except UnpicklingError as e:
            data = (None, CQueueRuntimeError("Host sent result that couldn't be unpickled: %s" % (e,)))
            label = 'result'

        if label == 'result':
            res, err = data

            if err:
                self.scheduler.task_done(hostid, monotime())

            self.results.append((self.hosts[hostid].user_host, index, res, err))
            self._notify()
        elif label == 'rpc' or (isinstance(label, tuple) and label[0] == 'rpc'):
            self.rpc_dispatcher.dispatch(addr, data)

    @handle('part_request')
    def __part_request(self, msg, data, path, hostid, iface):
        path = fix_path(msg, path)
        hostid, host = path[0]
        self.log.info('%s requested for %s', msg['nextsn'], hostid)

        if hostid not in self.scheduler.status:
            self.log.debug('ignored response from %s', host)
            return

        self.scheduler.part_requested(hostid, msg['nextsn'], monotime())

    def _orphan_timeout(self):
        return max(
            cfg.client.MinOrphanTimeout,
            cfg.client.Heartbeats.OrphanTimeoutMultiplier * self.scheduler.heartbeat_period,
            self._task_opts.get('orphan_timeout', 0)
        )

    def __del__(self):
        try:
            del self._event_fd
        finally:
            del self._event_notify_fd


class MultiTaskSession(Session):
    def __init__(self,
                 client,
                 signer,
                 hosts,
                 runnable,
                 task_hosts_data=None,
                 task_type='multi_task',
                 task_opts=None,
                 log=None,
                 ):
        super(MultiTaskSession, self).__init__(
            client,
            signer,
            hosts,
            runnable,
            task_hosts_data=task_hosts_data,
            task_type=task_type,
            task_opts=task_opts,
            log=log,
        )
        self.multipart_size = 0  # FIXME multipart is not supported in multitask yet
        self.scheduler.task_parts = 1
        self._task_parts = None

    @classmethod
    def prepare_host_data(cls, hosts, default_port, task_hosts_data):
        hosts_ = {}
        user_hosts = {}
        task_hosts_data_ = {}
        if task_hosts_data is None:
            task_hosts_data = [None] * len(hosts)
        for (hostid, (host, cont)), data in zip(enumerate(hosts), task_hosts_data):
            h = hosts_[hostid] = HostEntry(host, default_port)
            h.user_host = (host, cont)
            user_hosts[h.user_host] = hostid
            task_hosts_data_[hostid] = (cont, data)
        return hosts_, user_hosts, task_hosts_data_

    def schedule(self):
        heartbeats, hb_msg, task_msgs, stop_hosts, stop_message = super(MultiTaskSession, self).schedule()
        msgs = []
        for task_hosts, task_msg, task_host_data in task_msgs:
            task_hosts, task_host_data = self.aggregate_task(task_hosts, task_host_data)
            msgs.append((task_hosts, task_msg, task_host_data))
        return heartbeats, hb_msg, msgs, stop_hosts, stop_message

    @classmethod
    def aggregate_task(cls, hosts, data):
        hosts_map = defaultdict(set)
        new_hosts = []
        new_data = {}

        for hostid, addr in hosts:
            hosts_map[addr].add(hostid)

        for addr, ids in six.iteritems(hosts_map):
            hostids = tuple(ids)
            new_data[hostids] = {hostid: data[hostid] for hostid in ids} if data is not None else None
            new_hosts.append((hostids, addr))

        return new_hosts, new_data

    def _check_result(self, initiator, msg):
        hostids, host = initiator
        if not isinstance(hostids, (tuple, list)):
            hostids = (hostids,)

        for hostid in hostids:
            if hostid not in self.scheduler.status:
                self.log.debug('ignored result from %s [%s]', host, hostid)
                continue

            for n, data in self.scheduler.result_received(hostid, msg['index'], msg['result'], monotime()):
                self.handle_result(hostid, n, data)

    @handle('response')
    def __response(self, msg, data, path, hostids, iface):
        path = fix_path(msg, path)
        self.__update_port(path[0])

        content = msg['content']
        index = msg['index']

        hostids, host = path[0]
        if not isinstance(hostids, (tuple, list)):
            hostids = (hostids,)

        inner_type = content['type']

        try:
            inner_msg = self.loads(content['data'])
        except UnpicklingError as e:
            err = CQueueRuntimeError("Host sent result that couldn't be unpickled: %s" % (e,))
            for hostid in hostids:
                if hostid not in self.scheduler.status:
                    self.log.debug('ignored response from (%s, %s)', hostid, host)
                    continue

                self.scheduler.task_done(hostid, monotime())
                self.results.append((self.hosts[hostid].user_host, index, None, err))
                self._notify()
            return

        for hostid in hostids:
            if hostid not in self.scheduler.status:
                self.log.debug('ignored response from (%s, %s)', hostid, host)
                continue

            self.log.debug('response/%s #%s from (%s, %s)', inner_type, index, hostid, host)

            for _, label, index, data in self.scheduler.response_received(hostid,
                                                                          index,
                                                                          inner_type,
                                                                          inner_msg,
                                                                          monotime()):
                if label == 'result':
                    res, err = data

                    if err:
                        self.scheduler.task_done(hostid, monotime())

                    self.results.append((self.hosts[hostid].user_host, index, res, err))
                    self._notify()
                elif label == 'rpc' or (isinstance(label, tuple) and label[0] == 'rpc'):
                    self.rpc_dispatcher.dispatch((hostid, host), data)


class PortoshellSession(MultiTaskSession):
    @classmethod
    def prepare_host_data(cls, hosts, default_port, task_hosts_data):
        hosts_ = {}
        user_hosts = {}
        task_hosts_data_ = {}
        if task_hosts_data is None:
            task_hosts_data = [None] * len(hosts)
        for (hostid, item), data in zip(enumerate(hosts), task_hosts_data):
            h = hosts_[hostid] = HostEntry(item[0], default_port)
            h.user_host = item
            user_hosts[h.user_host] = hostid
            task_hosts_data_[hostid] = data
        return hosts_, user_hosts, task_hosts_data_


def fix_path(msg, path):
    aggr_path = msg.get('aggr_path')
    return aggr_path or path


class HostEntry(object):
    __slots__ = [
        'user_host',  # host as it was provided by user, never changes
        'hostname',  # hostname part of the host
        'base_port',  # server initial port
        'updated_port',  # task port
    ]

    def __init__(self, host, default_port):
        self.user_host = host
        self.hostname, self.base_port = self.split_host(host, default_port)
        self.updated_port = self.base_port

    @property
    def actual_addr(self):
        return self.hostname, self.updated_port

    @property
    def original_addr(self):
        return self.hostname, self.base_port

    @staticmethod
    def split_host(host, default_port):
        if isinstance(host, tuple):
            return str(host[0]), host[1]

        host = str(host)
        br_start = host.find('[')
        br_end = host.find(']')
        if 0 <= br_start <= br_end:
            # ipv6 address
            port_start = host.find(':', br_end + 1)
            if port_start > br_end:
                return host[br_start + 1:br_end], int(host[port_start + 1:])
            else:
                return host[br_start + 1:br_end], default_port
        else:
            host = host.split(':')
            if len(host) > 1:
                return host[0], int(host[1])
            else:
                return host[0], default_port


class Timeout(CQueueRuntimeError):
    def __init__(self, message='Timeout'):
        super(Timeout, self).__init__("Timeout: {}".format(message))

    def __str__(self):
        return str(self.message)


class LogHandler(RPCObject):
    def __init__(self):
        super(LogHandler, self).__init__()
        self.windows = defaultdict(IncomingWindow)

    def process(self, hostid, addr, msg):
        data = msg
        data['host'] = addr

        idx = data.pop('__log_index')
        uuid = data.pop('__log_uuid')

        window = self.windows[uuid]
        window.put(idx, data)

        for _, data in window.pop():
            record = logging.makeLogRecord(data)
            logging.getLogger(data['name']).handle(record)


class SshHandler(RPCObject):
    def __init__(self, taskid, signer, route, log):
        super(SshHandler, self).__init__()
        self.windows = defaultdict(IncomingWindow)
        self.taskid = taskid
        self.signer = signer
        self.route = route
        self.log = log

    def process(self, hostid, addr, data):
        idx = data['index']
        uuid = data['uuid']

        window = self.windows[uuid]
        window.put(idx, data)

        for idx, data in window.pop():
            req_type = data['type']
            uuid = data['uuid']
            if req_type == 'request_keys':
                # TODO think of merging multiple requests into one
                keys = iter(self.signer)
                response = response_keys(uuid, idx, keys, self.taskid)
                self.route(response, [(hostid, addr)])
            elif req_type == 'request_signs':
                signs = self.signer.sign(data['hash'], data['fingerprints'])
                response = response_signs(uuid, idx, signs, self.taskid)
                self.route(response, [(hostid, addr)])
            else:
                self.log.info('ignored sshagent request from host %s: unknown type %s',
                              short(self.taskid),
                              addr,
                              req_type)


def heartbeat(taskid):
    return {
        'uuid': genuuid(),
        'taskid': taskid,
        'next': 0,  # backward compatibility
        'time': time.time(),
        'type': 'heartbeat',
    }


def stop_msg(taskid):
    return {
        'uuid': genuuid(),
        'taskid': taskid,
        'time': time.time(),
        'type': 'stop',
    }


def rpc_msg(uuid, rpctype, rpcid, data):
    return {
        'uuid': genuuid(),
        'taskid': uuid,
        'data': data,
        'type': 'rpc',
        'rpctype': rpctype,
        'rpcid': rpcid,
    }


def response_keys(uuid, index, keys, taskid):
    return {
        'type': 'rpc',
        'rpctype': 'sshagent',
        'rpcid': None,
        'taskid': taskid,
        'uuid': genuuid(),
        'data': {
            'uuid': uuid,
            'index': index,
            'keys': [(key.publicKey().networkRepresentation(),
                      list(key.userNames),
                      key.comment) for key in keys],
        },
    }


def response_signs(uuid, index, signs, taskid):
    return {
        'type': 'rpc',
        'rpctype': 'sshagent',
        'rpcid': uuid,
        'taskid': taskid,
        'uuid': genuuid(),
        'data': {
            'index': index,
            'signatures': [(fp, sign) for fp, sign in signs],
        }
    }


def heartbeat_msg(taskid, **kwargs):
    return {
        'uuid': genuuid(),
        'taskid': taskid,
        'next': 0,
        'time': time.time(),
        'type': 'heartbeat',
        'params': kwargs
    }


def multipart_msg(data, sn, total, taskid):
    return {
        'uuid': genuuid(),
        'taskid': taskid,
        'sn': sn,
        'total': total,
        'data': data,
        'type': 'multipart',
    }
