from __future__ import absolute_import

import sys
import importlib
from collections import Callable

from ya.skynet.util.pickle import UnpicklingError


WHITE_MODULES = {
    'ya.skynet.services.cqudp.exceptions',
    'exceptions',
    'api.copier.errors',
    'skybone.rpc.errors',
}
WHITE_ATTRS = {
    'ya.skynet.services.cqudp.server.processhandle': {'CommunicationError', 'Signalled'},
    'ya.skynet.services.cqudp.rpc.pipe': {'Empty'},
    'ya.skynet.util.errors': {'_makeError'},
    'kernel.util.errors': {'_makeError'},
    'os': {'_make_stat_result'},
    '__builtin__': {'set'},
    'collections': {'OrderedDict', 'deque', 'Counter', 'defaultdict'},
}


def check_in_whitelist(module_name, attr_name, obj, default_policy, additional_whitelist=None):
    if module_name in WHITE_MODULES:
        return True

    if attr_name in WHITE_ATTRS.get(module_name, ()):
        return True

    if additional_whitelist and attr_name in additional_whitelist.get(module_name, ()):
        return True

    return getattr(obj, 'allow_unpickle', default_policy)


def import_if_permitted(module_name, attr_name, default_policy, modules=sys.modules, additional_whitelist=None):
    if not module_name:
        raise UnpicklingError("cannot lookup (%r %r) in non-module" % (module_name, attr_name))

    module = modules.get(module_name)
    if module is None:
        try:
            module = importlib.import_module(module_name)
        except ImportError as e:
            raise UnpicklingError("cannot import module %r required for object type %r: %s" % (module_name, attr_name, e))

    obj = getattr(module, attr_name)

    if not isinstance(obj, Callable):
        # anything non-callable is permitted
        return obj

    if not check_in_whitelist(module_name, attr_name, obj, default_policy, additional_whitelist=additional_whitelist):
        raise UnpicklingError("Import of ({0!r}.{1!r}) is forbidden. Use `cqudp_client.register_safe_unpickle({0!r}, {1!r})` to allow this type to be returned.".format(module_name, attr_name))

    return obj
