from __future__ import print_function, absolute_import

from ya.skynet.services.cqudp import cfg
from ya.skynet.services.cqudp.transport import protocol, envelope
from ya.skynet.services.cqudp.utils import configure_log, genuuid

import sys
import socket
import argparse


def ping_msg():
    return {
        'uuid': genuuid(),
        'type': 'bustest',
    }


def make_parser(root_parser=None):
    if root_parser is None:
        parser = argparse.ArgumentParser(description='Check cqudp bus connectivity with remote server')
    else:
        parser = root_parser.add_parser('buscheck', help='Check cqudp bus connectivity with remote_server')

    parser.add_argument('-i', '--impl',
                        type=str,
                        choices=('netlibus', 'msgpack'),
                        default='netlibus',
                        help='choose bus implementation to test')
    parser.add_argument('-p', '--port', type=int, default=None, help='non-default port remote bus is listening to')
    parser.add_argument('--check-ips',
                        action='store_true',
                        dest='check_ips',
                        default=False,
                        help='check each local interface')
    parser.add_argument('-v', '--verbose', action='store_true', help='display internal bus messages')
    parser.add_argument('host', type=str, help='remote hostname or IP to test')
    return parser


def get_port(impl, default_port):
    port = default_port

    if default_port:
        pass
    elif impl == 'netlibus':
        port = cfg.server.netlibus_port
    elif impl == 'msgpack':
        port = cfg.server.bus_port_msgpack

    if not port:
        print("Implementation is not available at this machine: %s" % impl, file=sys.stderr)
        sys.exit(2)

    return port


def try_connection(proto, host, port, interface):
    tree = [(
        (0, (host, port)),
        [((1, proto.listenaddr()), None)]
    )]
    msg = ping_msg()
    env = envelope.Envelope(msg, tree)
    for addr, next_env in env.next():
        proto.route(next_env, addr)

    cmd, env, iface = proto.receive(timeout=5.0)

    if cmd != 'route':
        print("%s -> (%s, %s): got wrong cmd (`route` expected): %s" % (interface, host, port, cmd), file=sys.stderr)
        return False

    if env.msgs[0]['uuid'] != msg['uuid']:
        print("%s -> (%s, %s): response uuids doesn't match" % (interface, host, port), file=sys.stderr)
        return False

    print("%s -> (%s, %s) is OK" % (interface, host, port))
    return True


def main(args=None):
    args = args or make_parser().parse_args()
    if args.verbose:
        configure_log('buscheck', 'DEBUG', True)

    port = get_port(args.impl, args.port)
    host = args.host

    proto = protocol.Protocol(impl=args.impl)
    if not args.check_ips:
        if not try_connection(proto, host, port, proto.listenaddr()):
            sys.exit(1)

    else:
        from ya.skynet.util.net.getifaddrs import getIfAddrs, Flags
        from ya.skynet.util.net.socketserver import getValidAddrs

        addrs = getValidAddrs((socket.AF_INET, socket.AF_INET6), getIfAddrs())
        addrs = [a.addr
                 for a in addrs
                 if a.flags & Flags.IFF_LOOPBACK == 0
                 ]

        if_port = proto.listenaddr()[1]

        if not all(try_connection(proto, host, port, (addr, if_port)) for addr in addrs):
            sys.exit(1)


if __name__ == '__main__':
    main()
