from __future__ import print_function, absolute_import

from ya.skynet.services.cqudp import cfg
from ya.skynet.services.cqudp.transport import protocol, envelope
from ya.skynet.services.cqudp.utils import configure_log, genuuid
from ya.skynet.services.cqudp.debugtools.reverse import PyrasiteIPC
from ya.skynet.library.auth.sign import FileKeysSignManager, ChainSignManager
from ya.skynet.library.auth.signagent import SignAgentClient

import six
import hashlib
import sys
import time
import socket
import argparse

from ya.skynet.util.sys.user import getUserName
from ya.skynet.util.pickle import loads
from ya.skynet.util.errors import formatException


def make_parser(root_parser=None):
    if root_parser is None:
        parser = argparse.ArgumentParser(description='Call cqudp server debug methods')
    else:
        parser = root_parser.add_parser('debug-requests', help='Call cqudp server debug methods')

    parser.add_argument('-i', '--impl',
                        type=str,
                        choices=('netlibus', 'msgpack'),
                        default='netlibus',
                        help='choose bus implementation to test')
    parser.add_argument('-p', '--port', type=int, default=None, help='non-default port remote bus is listening to')
    parser.add_argument('-P', '--service-port', default=12345, help='port to start service on')
    parser.add_argument('-v', '--verbose', action='store_true', help='display internal bus messages')
    parser.add_argument('host', type=str, help='remote hostname or IP to test')
    parser.add_argument('service', type=str, choices=('dowser', 'pyrasite', 'yappi'), help='debug service to operate on')
    parser.add_argument('action', type=str, choices=('start', 'stop'), help='debug command to execute')
    return parser


def get_port(impl, default_port):
    port = default_port

    if default_port:
        pass
    elif impl == 'netlibus':
        port = cfg.server.netlibus_port
    elif impl == 'msgpack':
        port = cfg.server.bus_port_msgpack

    if not port:
        print("Implementation is not available at this machine: %s" % impl, file=sys.stderr)
        sys.exit(2)

    return port


def request(service, action, port=None):
    return {
        'uuid': genuuid(),
        'type': 'debug_request',
        'service': service,
        'action': action,
        'signs': [],
        'acc_user': getUserName(),
        'acc_host': socket.getfqdn(),
        'ctime': time.time(),
        'port': port,
    }


def hash_request(task):
    md5 = hashlib.md5(six.b(task['uuid']))
    md5.update(six.b(task['acc_user']))
    md5.update(six.b(task['acc_host']))
    md5.update(six.b(task['service']))
    md5.update(six.b(task['action']))
    md5.update(six.b(str(task['port'])))
    md5.update(six.b(str(task['ctime'])))
    return md5.digest()


def pyrasite_interact(ipc):
    ipc.wait()

    prompt, payload = ipc.recv().split('\n', 1)
    print(payload)

    try:
        import readline  # noqa
    except ImportError:
        pass

    # py3k compat
    try:
        input_ = raw_input
    except NameError:
        input_ = input

    try:
        while True:
            try:
                input_line = input_(prompt)
            except EOFError:
                input_line = 'exit()'
                print('')
            except KeyboardInterrupt:
                input_line = 'None'
                print('')

            ipc.send(input_line)
            payload = ipc.recv()
            if payload is None:
                break
            prompt, payload = payload.split('\n', 1)
            if payload != '':
                print(payload)
    except:
        print('')
        raise
    finally:
        ipc.close()


def main(args=None):
    args = args or make_parser().parse_args()
    if args.verbose:
        configure_log('debugtools', 'DEBUG', True)

    port = get_port(args.impl, args.port)
    host = args.host

    fksm = FileKeysSignManager(
        userKeyDirs=cfg.client.Auth.UserKeyDirs,
        keyFiles=cfg.client.Auth.KeyFiles,
    )
    fksm.load()
    sa = SignAgentClient()
    sa.load()
    sm = ChainSignManager([fksm, sa])

    proto = protocol.Protocol(impl=args.impl)

    if args.service == 'pyrasite':
        ipc = PyrasiteIPC()
        ipc.hostname = '::'
        args.service_port = int(args.service_port)
        ipc.port = args.service_port
        ipc.listen()
        print("bound to [%s]:%s" % (ipc.hostname, ipc.port))
    elif args.service == 'yappi':
        args.service_port = '/tmp/cqudp.callgrind' if isinstance(args.service_port, int) else args.service_port

    tree = envelope.build_tree([(0, (host, port))])
    msg = request(args.service, args.action, args.service_port)
    msg['signs'].extend(sm.sign(hash_request(msg)))
    env = envelope.Envelope(msg, tree)
    env.path.insert(0, ('C', proto.listenaddr()))
    for addr, next_env in env.next():
        proto.route(next_env, addr)

    try:
        cmd, env, iface = proto.receive(timeout=10.0)
    except proto.Timeout:
        print("(%s, %s): receive timeout" % (host, port), file=sys.stderr)
        sys.exit(1)

    assert cmd == 'route', "(%s, %s): got wrong cmd (`route` expected): %s" % (host, port, cmd)

    resp = env.msgs[0]
    assert resp['request'] == msg['uuid'], "(%s, %s): response uuids doesn't match" % (host, port)

    if resp['result']:
        msg = resp['result']
        if resp['port']:
            msg += ' (port %s)' % (resp['port'],)
        print(msg)
    elif resp['error']:
        print("call finished with error:", file=sys.stderr)
        print(formatException(loads(resp['error'])), file=sys.stderr)
        sys.exit(1)
    else:
        print("unexpected response:", file=sys.stderr)
        print(resp, file=sys.stderr)
        sys.exit(1)

    if args.service == 'pyrasite':
        pyrasite_interact(ipc)


if __name__ == '__main__':
    main()
