from __future__ import print_function, absolute_import

from ya.skynet.services.cqudp import cfg
from ya.skynet.services.cqudp.transport import protocol, envelope
from ya.skynet.services.cqudp.utils import configure_log, genuuid

import sys
import json
import argparse


def stats_msg():
    return {
        'uuid': genuuid(),
        'type': 'request_stats',
    }


def make_parser(root_parser=None):
    if root_parser is None:
        parser = argparse.ArgumentParser(description='Get cqudp task stats')
    else:
        parser = root_parser.add_parser('stats', help='Get cqudp task stats')

    parser.add_argument('-i', '--impl',
                        type=str,
                        choices=('netlibus', 'msgpack'),
                        default='netlibus',
                        help='choose bus implementation to test')
    parser.add_argument('-p', '--port', type=int, default=None, help='non-default port remote bus is listening to')
    parser.add_argument('-v', '--verbose', action='store_true', help='display internal bus messages')
    return parser


def get_port(impl, default_port):
    port = default_port

    if default_port:
        pass
    elif impl == 'netlibus':
        port = cfg.server.netlibus_port
    elif impl == 'msgpack':
        port = cfg.server.bus_port_msgpack

    if not port:
        print("Implementation is not available at this machine: %s" % impl, file=sys.stderr)
        sys.exit(2)

    return port


def print_stats(proto, port):
    tree = envelope.build_tree([(0, ('localhost', port))])
    msg = stats_msg()
    env = envelope.Envelope(msg, tree)
    env.path.insert(0, ('C', proto.listenaddr()))
    for addr, next_env in env.next():
        proto.route(next_env, addr)

    cmd, env, iface = proto.receive(timeout=5.0)

    if cmd != 'route':
        print("got wrong cmd (`route` expected): %s" % (cmd,), file=sys.stderr)
        return False

    if env.msgs[0]['type'] != 'stats':
        print("response type doesn't match (`stats` expected): %s" % (env.msgs[0]['type'],), file=sys.stderr)
        return False

    print(json.dumps(env.msgs[0]['by_user']))
    return True


def main(args=None):
    args = args or make_parser().parse_args()
    if args.verbose:
        configure_log('stats', 'DEBUG', True)

    port = get_port(args.impl, args.port)

    proto = protocol.Protocol(impl=args.impl)
    if not print_stats(proto, port):
        sys.exit(1)


if __name__ == '__main__':
    main()
