import os
import imp
import sys
import types
import importlib
try:
    import pkg_resources
except ImportError:
    pkg_resources = None

from ya.skynet.util.functional import cached
from ya.skynet.services.cqudp.utils import log as root


excludedPackages = [
    'kernel',
    'api',
    'srvmngr',
    'cqudp',
    'kqueue',
    'cqueue',
    'ya',  # temporary hack, we just mustn't package namespace packages
    'msgpack',
    'Crypto',
    'execnet',  # from pytest
    'pkg_resources',
    'skynet_logger',
]

excludedPrefixes = [
    'ya.skynet.util',
]


excludedPath = [
    os.path.dirname(os.__file__),
    os.path.dirname(os.path.abspath(os.__file__)),
]
if hasattr(sys, 'real_prefix'):  # python 2.7 venv
    excludedPath.append(os.path.join(sys.real_prefix, 'lib/python2.7'))
elif hasattr(sys, 'base_exec_prefix'):  # python 3 venv
    excludedPath.append(os.path.join(sys.base_exec_prefix, 'lib/python2.7'))


@cached(60)
def why_excluded_module(name):
    module = sys.modules.get(name)

    if not module:
        return 'NOT IN SYS'

    elif not getattr(module, '__name__', None):
        return 'HAS NO NAME'

    elif name != '__main__' and (imp.is_builtin(module.__name__) or not hasattr(module, "__file__")):
        return 'BUILTIN'

    elif any(name.startswith(prefix) for prefix in excludedPrefixes):
        return 'EXCLUDED BY PREFIX'

    elif name != '__main__' and not getattr(module, '__file__', None):
        return 'ARTIFICIAL MODULE'

    if name != '__main__' and hasattr(module, '__file__'):
        file1 = module.__file__
        file2 = None

        if (
            file1.endswith('.pyd') or
            file1.endswith('.pyo') or
            file1.endswith('.dll') or
            file1.endswith('.so') or
            file1.endswith('.dylib')  # practically dylib shouldn't be used anyway
        ):
            return 'BINARY MODULE'

        for path in excludedPath:
            if file1.startswith(path):
                return 'EXCLUDED PATH {}'.format(file1)
            if file2 is None:
                file2 = os.path.abspath(file1)
            if file2.startswith(path):
                return 'EXCLUDED ABSPATH {}'.format(file2)

    for package in excludedPackages:
        if module.__name__ == package or module.__name__.startswith(package + "."):
            return 'EXCLUDED PACKAGE {}'.format(package)
    return False


@cached(60)
def is_package(module):
    loader = getattr(module, '__loader__', None)
    if loader is not None and hasattr(loader, 'is_package'):
        try:
            return loader.is_package(module.__name__)
        except ImportError:
            pass

    # six makes you think it's a package and hence forces you to package all the
    # ~/.local/lib/pythonX.X/site-packages/ as its subpackages
    return getattr(module, '__path__', None) is not None and module.__name__ != 'six'


def _module_parent(fullname):
    return '.'.join(fullname.split('.')[:-1])


def is_namespace_package(fullname):
    return pkg_resources is not None and fullname in pkg_resources._namespace_packages


def add_namespace_children(name, module, modules_dict):
    if pkg_resources is not None and name in pkg_resources._namespace_packages:
        # This package is a namespace package, so we need to add namespace
        # package itself several times to modules list.
        sys_modules = list(sys.modules.values())
        for sub_name in dir(module):
            submod = getattr(module, sub_name)
            if (
                isinstance(submod, types.ModuleType) and
                submod in sys_modules
            ):
                modules_dict[submod.__name__] = submod
                if is_package(submod):
                    add_namespace_children(submod.__name__, submod, modules_dict)


def resolve_relative(path):
    if not os.path.isabs(path):
        for sysPath in sys.path:
            if os.path.exists(os.path.join(sysPath, path)):
                return os.path.join(sysPath, path)
    return path


def collect_modules(modules_list=None, log=None):
    """
    Collect modules info
    :param list modules_list: optional list of module names
    :rtype: set
    :return: set of tuples (dirpath, module_name, module_object, archive_filename)
             where:
                 dirpath - absolutepath to dir with module
                 module_name - str with module name
                 module_object - object of type :class:`types.ModuleType`, module or root module of package
                 archive_filename - new filename under which module is to be stored in archive, None for packages
    """
    log = log or root().getChild('eggs')
    modules = set()
    if modules_list is None:
        modules_dict = {name: module
                        for name, module in list(sys.modules.items())
                        if '.' not in name
                        or _module_parent(name) not in sys.modules
                        or is_namespace_package(_module_parent(name))}
    else:
        modules_dict = {name: importlib.import_module(name) for name in modules_list}

    for name, module in list(modules_dict.items()):
        excludedReason = why_excluded_module(name)
        if excludedReason is not False:
            log.debug('Module skipped because of {}: {}'.format(excludedReason, name))
            del modules_dict[name]
        elif is_package(module):
            add_namespace_children(name, module, modules_dict)

    for name, module in list(modules_dict.items()):
        if name == '__main__':
            continue  # it's processed separately
        directory, filename = os.path.split(module.__file__)
        fileExt = os.path.splitext(filename)[1]
        if is_package(module):
            # Dont be confused with namespace packages above -- if a package is a namespace
            # package, but currently only imported itself (without subpackages) -- this
            # line will take place. It will not change anything else with namespace packages
            # in other case.
            p = resolve_relative(directory)
            modules.add((p, name, module, None))
            log.debug("Packaging package: {}".format(p))
        else:
            p = resolve_relative(module.__file__)
            n = "%s%s" % (module.__name__, fileExt)
            modules.add((p, name, module, n))
            log.debug("Packaging module: {} as {}".format(p, n))

    if '__main__' in modules_dict:
        mainModule = sys.modules['__main__']
        mainModuleFile = os.path.basename(mainModule.__file__) if hasattr(mainModule, '__file__') else None
        fileExt = os.path.splitext(mainModuleFile)[1] if mainModuleFile else '.pyc'
        if not fileExt:
            fileExt = ".py"  # We assumes it for ext less files

        # dir with main module Always in sys path
        p = resolve_relative(mainModule.__file__) if hasattr(mainModule, '__file__') else None
        n = "__new__main__%s" % fileExt
        modules.add((p, '__new__main__' if p else '__main__', mainModule, n))
        log.debug("Packaging main: {} as {}".format(p, n))

    return modules
