import os
import marshal
import zipfile
import types
from zipimport import zipimporter

from ya.skynet.util.functional import cached
from ya.skynet.util import pickle

import six


def _package_files(name, dirpath):
    for filename in os.listdir(dirpath):
        path = os.path.join(dirpath, filename)
        if os.path.isdir(path):
            if (
                os.path.exists(os.path.join(path, "__init__.py")) or
                os.path.exists(os.path.join(path, "__init__.pyc"))
            ):
                for rec in _package_files('.'.join((name, filename)), path):
                    yield rec
        elif filename.endswith(".py"):
            yield dirpath, filename, name
        elif filename.endswith(".pyc") and not os.path.exists(os.path.join(dirpath, filename[:-1])):
            yield dirpath, filename, name


def _parent_namespace_files(name, dirname, namespace):
    parts = name.split('.')[:-1]
    dirname = os.path.dirname(dirname)  # parent dir belonging to the upper level

    while parts and '.'.join(parts).startswith(namespace):
        py_file = os.path.join(dirname, '__init__.py')
        pyc_file = os.path.join(dirname, '__init__.pyc')
        if os.path.exists(py_file):
            yield dirname, '__init__.py', os.path.sep.join(parts)
        elif os.path.exists(pyc_file):
            yield dirname, '__init__.pyc', os.path.sep.join(parts)
        else:
            break  # This situation should never be reached, really

        parts.pop()
        dirname = os.path.dirname(dirname)


@cached(60)
def get_package_sources(module):
    return list(_get_package_sources(module))


def _get_package_sources(module):
    loader = getattr(module, '__loader__', None)
    if loader is not None and hasattr(loader, 'package_submodules'):
        # NOTE: feature is not defined in PEP-0302 or anywhere else
        # because python is a piece of shit. But we defined it as an
        # extension to our own importer.
        submodules = [module.__name__] + loader.package_submodules(module.__name__)

        for mod in submodules:
            try:
                src = loader.get_source(mod)
            except ImportError:
                src = None
            try:
                code = loader.get_code(mod)
            except ImportError:
                code = None
            basename = mod.replace('.', os.path.sep)
            if loader.is_package(mod):
                basename = os.path.join(basename, '__init__')
            yield (code.co_filename if code else None), basename, src, code

    elif isinstance(loader, zipimporter):
        submodules = set([module.__name__])
        basename = os.path.join(module.__name__.replace('.', os.path.sep), '__init__')
        code = get_code(module)
        fullpath = code and code.co_filename
        fullpath = fullpath or (os.path.join(loader.archive, basename) + ('.py' if src else '.pyc'))
        yield fullpath, basename, get_source(module), code

        zf = zipfile.ZipFile(loader.archive)
        prefix = module.__name__ + '.'
        for filename in six.iterkeys(loader._files):
            basename, ext = os.path.splitext(filename)
            parts = basename.split(os.path.sep)
            if parts[-1] == '__init__':
                parts.pop()
            submod = '.'.join(parts)
            if submod in submodules:
                continue
            elif submod.startswith(prefix):
                if ext == '.py':
                    src = zf.read(filename)
                    code = zf.read(filename + 'c') if (filename + 'c') in loader._files else None
                elif ext == '.pyc':
                    code = zf.read(filename)
                    src = zf.read(filename[:-1]) if (filename[:-1]) in loader._files else None
                else:
                    continue
                code = marshal.loads(code[8:]) if code is not None else None
                fullpath = code and code.co_filename
                fullpath = fullpath or os.path.join(loader.archive, basename) + ('.py' if src else '.pyc')
                yield fullpath, basename, src, code
                submodules.add(submod)
    else:
        submodules = set()
        for dirpath, filename, basename in _package_files(module.__name__, os.path.dirname(module.__file__)):
            fn = os.path.join(basename.replace('.', os.path.sep), os.path.splitext(filename)[0])
            if filename.endswith('.py'):
                fullpath = os.path.join(dirpath, filename)
                yield fullpath, fn, open(fullpath).read(), None  # don't believe to fs-stored .pyc
            elif filename.endswith('.pyc'):
                try:
                    fullpath = os.path.join(dirpath, filename)
                    yield fullpath, fn, None, marshal.loads(open(fullpath, 'rb').read()[8:])
                except ValueError:  # incompatible pyc version
                    pass


def get_in_memory_source(name, module):
    sort_order = {
        types.ModuleType: 0,
        types.FunctionType: 1,
        types.ClassType: 1,
        type: 1,
    }
    source = six.StringIO()
    for key, value in sorted(six.iteritems(module.__dict__), key=lambda item: sort_order.get(type(item[1]), 2)):
        if key in (
            'In',
            'Out',
            '_',
            '__',
            '__builtin__',
            '___',
            '_dh',
            '_i',
            '_ih',
            '_ii',
            '_iii',
            '_oh',
            '_sh',
            'exit',
            'get_ipython',
            'help',
            'quit',
            'execmodel',
        ):
            continue  # omit ipython generated stuff

        _write_in_memory_item(source, name, '', key, value)

    return source.getvalue()


def _write_in_memory_item(stream, basemod, indent, name, item):
    if isinstance(item, six.string_types + six.integer_types + (six.binary_type, types.NoneType, types.FloatType,)):
        stream.write(indent)  # %s = %r
        stream.write(name)
        stream.write(' = ')
        stream.write(repr(item))
        stream.write('\n')
    elif isinstance(item, types.ModuleType):
        stream.write(indent)
        stream.write(name)
        stream.write(' = __import__("importlib").import_module(')
        stream.write(repr(item.__name__))
        stream.write(')\n')
    elif isinstance(item, (types.FunctionType, types.ClassType, type)) and item.__module__ is not None and item.__module__ != basemod:
        stream.write(indent)
        stream.write("from ")
        stream.write(item.__module__)
        stream.write(" import ")
        stream.write(item.__name__)
        stream.write(" as ")
        stream.write(name)
        stream.write("\n")
    elif isinstance(item, (types.FunctionType, types.MethodType, staticmethod, classmethod)):
        marker = None
        if isinstance(item, (staticmethod, classmethod)):
            marker = type(item).__name__
            f = item.__func__
        f = item.__func__ if hasattr(item, '__func__') else item
        code = f.func_code
        defs = f.func_defaults

        stream.write(indent)  # %s = types.FunctionType(code, globals, name, argdefs)
        stream.write(name)
        stream.write(' = ')
        if marker:
            stream.write(marker)
        stream.write('(__import__("types").FunctionType(__import__("marshal").loads(')
        stream.write(repr(marshal.dumps(code)))
        stream.write('), globals(), ')
        stream.write(repr(name))
        stream.write(', __import__("cPickle").loads(')
        stream.write(repr(pickle.dumps(defs)))
        stream.write(')))\n')
    elif isinstance(item, (types.ClassType, type)):
        mro = getattr(item, '__mro__', None)
        mro = list(mro) if mro else None
        if mro and item in mro:
            mro.remove(item)
        stream.write(indent)
        stream.write("class ")
        stream.write(name)
        if mro:
            stream.write("(")
            stream.write(", ".join(t.__name__ for t in mro))
            stream.write(")")
        stream.write(":\n")
        indent += '    '
        for k, v in six.iteritems(item.__dict__):
            if k in ('__module__', '__dict__', '__weakref__',):
                continue
            _write_in_memory_item(stream, basemod, indent, k, v)
    else:
        stream.write(indent)
        stream.write(name)
        stream.write(' = __import__("cPickle").loads(')
        stream.write(repr(pickle.dumps(item)))
        stream.write(')\n')


def _read_file(path, binary=False):
    mode = 'rb' if binary else 'r'
    if os.path.isfile(path):
        with open(path, mode) as f:
            return f.read()

    head, tail = os.path.split(path)
    while head.lstrip('/') != '':
        if os.path.isfile(head):
            try:
                reader = zipfile.ZipFile(head, compression=zipfile.ZIP_DEFLATED)
                with reader.open(tail, mode) as f:
                    return f.read()
            except:
                return None
        head, new_tail = os.path.split(tail)
        tail = os.path.join(new_tail, tail)


@cached(60)
def get_source(module):
    loader = getattr(module, '__loader__', None)
    if loader is not None and hasattr(loader, 'get_source'):
        try:
            return loader.get_source(module.__name__)
        except ImportError:
            pass

    base, ext = os.path.splitext(module.__file__)
    filename = base + '.py'
    result = _read_file(filename, binary=False)
    if result is None and getattr(module, '__name__', None) == '__main__':
        return _read_file(module.__file__)


@cached(60)
def get_code(module):
    loader = getattr(module, '__loader__', None)
    if loader is not None and hasattr(loader, 'get_code'):
        try:
            return loader.get_code(module.__name__)
        except ImportError:
            pass

    filename = os.path.splitext(module.__file__)[0] + '.pyc'
    data = _read_file(filename, binary=True)
    if data is not None:
        try:
            return marshal.loads(data[8:])
        except ValueError:  # wrong pyc version
            pass
