from __future__ import absolute_import

import os
import re
import zipfile

from .collect import is_package, is_namespace_package
from .retrieve import get_package_sources, get_source, get_code, get_in_memory_source
from .compiled_file import InMemoryCompiledFile

from ya.skynet.services.cqudp.exceptions import CQueueRuntimeError
from ya.skynet.services.cqudp.utils import log as root

import six


MAIN_STUB = re.compile(
    '^if[\s]*__name__[\s]*==[\s]*[\'\""][\s]*__main__[\s]*[\'\""][\s]*:[\s]*($|#.*$)',
    re.MULTILINE
)

NS_PKG_STUB = '__import__("pkg_resources").declare_namespace(__name__)\n'


def _arch_ext(filename):
    result = os.path.splitext(filename)[1]
    if result:
        return result
    else:
        return '.py'


def check_main(src):
    return bool(MAIN_STUB.search(src))


def write_in_memory_main(egg, name, module, arcFile):
    source = get_in_memory_source(name, module)
    new_name = os.path.splitext(arcFile)[0].replace('.', os.path.sep)
    with InMemoryCompiledFile(name=new_name, original_file='<stdin>', original_source=source) as compiled:
        for ext, data in compiled:
            egg.writestr(new_name + ext, data)


def write_module(egg, path, module, arcFile):
    src = get_source(module)
    code = get_code(module)

    name = os.path.splitext(arcFile)[0].replace('.', os.path.sep)

    if name == '__new__main__' and src and not check_main(src):
        raise CQueueRuntimeError('Task cannot be run, would break the cluster: No `if __name__ == \'__main__\':` stub')

    with InMemoryCompiledFile(name=name, original_file=path, original_source=src, original_code=code) as compiled:
        for ext, data in compiled:
            egg.writestr(name + ext, data)


def write_package(egg, name, module, dirpath, namespace=None):
    # files = list(_package_files(name, dirpath))
    # if namespace is not None and namespace != name:
    #    files += list(_parent_namespace_files(name, dirpath, namespace))
    if is_namespace_package(name):
        basename = os.path.join(name.replace('.', os.path.sep), '__init__')
        path = os.path.join(dirpath, '__init__.py')
        with InMemoryCompiledFile(name=basename, original_file=path, original_source=NS_PKG_STUB) as compiled:
            for ext, data in compiled:
                egg.writestr(basename + ext, data)
        return

    prefix_to_remove = len(module.__name__)
    for fullpath, basename, module_source, module_code in get_package_sources(module):
        with InMemoryCompiledFile(name=basename, original_file=fullpath, original_source=module_source, original_code=module_code) as compiled:
            for ext, data in compiled:
                egg.writestr(name.replace('.', os.path.sep) + basename[prefix_to_remove:] + ext, data)


def package_egg(modules, log=None):
    log = log or root().getChild('eggs')
    # we could use six.BytesIO, but in py2 it creates slow StringIO
    if six.PY2:
        data = six.moves.cStringIO()
    else:
        data = six.BytesIO()
    writer = zipfile.ZipFile(data, "w", zipfile.ZIP_DEFLATED)

    for path, name, module, arcFile in modules:
        if path is None and name == '__main__':
            write_in_memory_main(writer, name, module, arcFile)
            log.debug('Writing in-memory %r as %r', name, arcFile)
        elif is_package(module):
            write_package(writer, name, module, path)
            log.debug("Writing %r", path)
        elif arcFile:
            write_module(writer, path, module, arcFile)
            log.debug("Writing %r as %r", path, arcFile)
        # else:
        #     tail, name = os.path.split(path)
        #     while tail.lstrip('/') != '':
        #         if os.path.isfile(tail):
        #             try:
        #                 reader = zipfile.PyZipFile(tail)
        #                 for nameInArc in reader.namelist():
        #                     if nameInArc.startswith(name):
        #                         # If there are many packages in egg file, we need only
        #                         # the ones we requested. This will also ignore other
        #                         # files/folders in egg (e.g. EGG-INFO/).
        #                         writer.writestr(nameInArc, reader.read(nameInArc))
        #                         log.debug('Writing `{}` from `{}`'.format(nameInArc, tail))
        #             except:
        #                 pass
        #             break
        #         tail, newName = os.path.split(tail)
        #         name = os.path.join(newName, name)

    writer.close()
    data.seek(0)
    return data.read()
