"""Internal exceptions that can be raised by cqudp."""

__author__ = 'torkve'

import os

from ya.skynet.util.errors import SkynetError


class CQueueError(SkynetError):
    """This is the base class for all CQueue exceptions."""
    pass


class CQueueNetworkError(CQueueError, EnvironmentError):
    """Error related to the network operations."""

    def __init__(self, *args, **kwargs):
        message = kwargs.pop('message')
        CQueueError.__init__(self, message, *args, **kwargs)

        errno = args[0] if len(args) else kwargs.get('errno')

        if len(args) > 1:
            strerror = args[1]
        elif 'strerror' in kwargs:
            strerror = kwargs.get('strerror')
        elif errno and isinstance(errno, int):
            strerror = os.strerror(errno)
        else:
            strerror = message

        if len(args) > 2 or 'filename' in kwargs:
            EnvironmentError.__init__(self, errno, strerror, args[2] if len(args) > 2 else kwargs.get('filename'))
        else:
            EnvironmentError.__init__(self, errno, strerror)


class CQueueRuntimeError(CQueueError, RuntimeError):
    """RuntimeError that occured in CQueue."""

    def __init__(self, message):
        CQueueError.__init__(self, message)
        RuntimeError.__init__(self, message)

    def __str__(self):
        return 'CQueue runtime error: {}'.format(self.message)


class CQueueExecutionError(CQueueError):
    """Error that has been raised during task execution."""

    def __init__(self, error):
        CQueueError.__init__(self, message="Task execution failed", error=error)


class CQueueAuthError(CQueueRuntimeError):
    """Base class for authentication and authorization problems"""

    def __str__(self):
        return str(self.message)


class CQueueAuthenticationError(CQueueAuthError):
    """Authentication failure"""
    def __init__(self, reason=None, user=None):
        self.user = user
        self.reason = reason
        super(CQueueAuthenticationError, self).__init__("Authentication as `{}` failed: {}".format(user, reason))


class CQueueAuthorizationError(CQueueAuthError):
    """Authorization failure"""
    def __init__(self, reason=None, user=None):
        self.user = user
        self.reason = reason
        super(CQueueAuthorizationError, self).__init__("Authorization as `{}` failed. PAM reported: {}".format(user, reason))


class CQueueExecutionFailure(CQueueRuntimeError):
    """Error with task, preventing it from start"""
    def __init__(self, message):
        super(CQueueExecutionFailure, self).__init__("Task cannot be started: {}".format(message))

    def __str__(self):
        return str(self.message)


class ReadableKeyError(CQueueRuntimeError):
    """Wrapper for KeyError with readable __str__()"""
    def __init__(self, e):
        super(ReadableKeyError, self).__init__("No such key: {}".format(e))

    def __str__(self):
        return self.message
