#!/skynet/python/bin/python

from __future__ import absolute_import, division, print_function

import os
import sys
import signal
import argparse
import tempfile

import six

from ya.skynet.util.errors import formatException

IN_ARCADIA = bool(getattr(sys, 'is_standalone_binary', False))

from ya.skynet.services.cqudp import cfg
from ya.skynet.services.cqudp.server.server_main import main as server_main
from ya.skynet.services.cqudp.server.task_main import main as task_main

if IN_ARCADIA:
    from ya.skynet.services.cqudp.scripts.start import arc_main as start_main
    from ya.skynet.services.cqudp.scripts.check import main as check_main
    from ya.skynet.services.cqudp.debugtools import buscheck, debugclient, stats

from ya.skynet.services.cqudp.utils import log


def dirtype(arg):
    os.listdir(arg)  # list dir to trigger every possible error
    return arg


def parse_args():
    parser = argparse.ArgumentParser(description='cqudp common entrypoint')
    subparsers = parser.add_subparsers(title='MODES', description='Execution modes', dest='mode')

    if IN_ARCADIA:
        parser_start = subparsers.add_parser('start-server', help='Helper to start a routing server')
        parser_start.add_argument('argv', nargs='*')
        parser_start.set_defaults(callable=start_main)

        parser_check = subparsers.add_parser('check-server', help='Check running server instance')
        parser_check.set_defaults(callable=check_main)

        parser_buscheck = buscheck.make_parser(subparsers)
        parser_buscheck.set_defaults(callable=buscheck.main)

        parser_debugclient = debugclient.make_parser(subparsers)
        parser_debugclient.set_defaults(callable=debugclient.main)

        parser_stats = stats.make_parser(subparsers)
        parser_stats.set_defaults(callable=stats.main)

    parser_server = subparsers.add_parser('server', help='Act as a routing server')
    parser_server.add_argument('--port-offset',
                               type=int,
                               dest='port_offset',
                               default=0,
                               help='Int offset to add to config port numbers')
    parser_server.add_argument('-m', '--msgpack-port',
                               dest='msgpack_port',
                               type=int,
                               default=cfg.server.bus_port_msgpack,
                               help='UDP msgpack port to bind on')
    parser_server.add_argument('-n', '--netlibus-port',
                               dest='netlibus_port',
                               type=int,
                               default=cfg.server.netlibus_port,
                               help='UDP netlibus port to bind on')
    parser_server.add_argument('-u', '--user', type=str, default=None, help='User to drop privileges to')
    parser_server.add_argument('-d', '--debug',
                               action='store_true',
                               default=False,
                               help='Log to terminal for debugging')
    parser_server.add_argument('-l', '--loglevel',
                               default=cfg.server.LogLevel,
                               choices=('DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL'),
                               help='log messages level')
    parser_server.add_argument('--yappi', action='store_true', default=None)
    parser_server.add_argument('-t', '--tempdir', type=dirtype, help='Custom temporary dir for tasks data')
    parser_server.add_argument('--taskpath', type=str, help='path used as path to task instead of argv[0]')
    parser_server.add_argument('--interpreter', type=str, help='path used as default interpreter for tasks instead of automatically deduced')
    parser_server.add_argument('--rc-socket', dest='rc_socket', help='UDS socket path or name for root operations')
    parser_server.add_argument('--watch-ppid',
                               action='store_true',
                               default=False,
                               dest='watch_ppid',
                               help='Die if ppid died')
    parser_server.add_argument('--bind-containers',
                               action='store_true',
                               default=cfg.server.bind_containers,
                               dest='bind_containers')
    parser_server.set_defaults(callable=server_main)

    parser_task = subparsers.add_parser('task', help='Act as a task execution node')
    parser_task.add_argument('taskid', type=str, help='Task UUID')
    parser_task.add_argument('hostid', type=int, help='This host ID')
    parser_task.add_argument('retpath', type=six.b, help='base64-encoded pickled task return path')
    parser_task.add_argument('task', type=str, help='path to file with pickled task')
    parser_task.add_argument('--notify',
                             type=six.b,
                             default=None,
                             help='base64-encoded path of local server to notify about task successful start')
    parser_task.add_argument('-d', '--debug',
                             action='store_true',
                             default=False,
                             help='Log to terminal for debugging')
    parser_task.add_argument('--log-level', dest='loglevel', type=str, default=None, help='Custom log level to use')
    parser_task.add_argument('-t', '--tempdir', type=dirtype, help='Custom temporary dir for tasks data')
    parser_task.add_argument('--cgroups-path',
                             type=str,
                             help='Base directory for cgroups',
                             dest='cgroups_path',
                             default='/sys/fs/cgroup')
    parser_task.add_argument('--cgroup', type=str, help='cgroup to move in')
    parser_task.add_argument('--user', type=str, help='user to drop privileges to')
    parser_task.add_argument('--chdir', type=str, help='dir to chdir to')
    parser_task.set_defaults(callable=task_main)

    return parser.parse_args()


def main():
    state_file = None

    try:
        args = parse_args()

        if 'tempdir' in args and args.tempdir:
            tempfile.tempdir = args.tempdir

        if args.mode == 'server':
            state_file = os.path.join(tempfile.gettempdir(), 'state', 'cqudp.state')
            server_main(state_file, args)
        else:
            args.callable(args)

    except KeyboardInterrupt:
        log().info('Caught SIGINT, stopping daemon')
        # The kostyl': disable further SIGINTs so that netliba would
        # have time to peacefully die and do not crash with segfault
        signal.signal(signal.SIGINT, signal.SIG_IGN)

    except SystemExit as ex:
        if ex.args:
            log().info(formatException())
        else:
            log().info('Got SystemExit exception in main loop')
        raise

    except Exception:
        log().info(formatException())
        raise SystemExit(1)
    finally:
        if state_file and os.path.exists(state_file):
            os.unlink(state_file)


if __name__ == '__main__':
    if os.getenv('COVERAGE_PROCESS_START', None):
        import coverage
        coverage.process_startup()

    main()
