"""
RPC-specific utility classes and functions.
"""
import struct
from ya.skynet.util.sys.gettime import monoTime


class Timer(object):
    """
    A simple scoped object which will calculate time amount which was spent within the scope the object
    was used with `with` statement.

    Also, it can automatically reduce the counter on block leave if it will be passed into the constructor.
    """

    def __init__(self, countdown=0):
        self.__countdown = countdown

    def __enter__(self):
        self.__start = monoTime()
        self.__finish = None
        return self

    def __exit__(self, type, value, traceback):
        # Error handling here
        self.__finish = monoTime()

    @property
    def counter(self):
        if self.__countdown is None:
            return None
        self.__countdown -= self.duration
        return self.__countdown

    @property
    def duration(self):
        return (monoTime() if not self.__finish else self.__finish) - self.__start


class ConnectionHandler(object):
    def __init__(self, magic, handshake_send_timeout, handshake_receive_timeout):
        self.__magic = magic
        self._handshake_send_timeout = handshake_send_timeout
        self._handshake_receive_timeout = handshake_receive_timeout
        assert 0 <= self.__magic < 2 ** 32

        super(ConnectionHandler, self).__init__()

    def get_magic(self):
        return struct.Struct('!I').pack(self.__magic)

    def handle(self, sock, sid=None):
        # Just send or receive the session ID given
        sid_packer = struct.Struct('!I')
        if sid:
            return sock.write(sid_packer.pack(sid), timeout=self._handshake_send_timeout)
        else:
            data = sock.read(sid_packer.size, timeout=self._handshake_receive_timeout)
            if data is None:
                return
            sid, = sid_packer.unpack(data)
            return sid


def sid2str(sid, bits=32):
    return hex(sid)[2:].rstrip('L').zfill(bits // 4).lower()
