import os
import fcntl
import errno
from collections import deque

from .utils import FdHolder, poll_select, monotime


class Selectable(object):
    def _get_event_fd(self):
        raise NotImplementedError

    def _is_data_ready(self):
        raise NotImplementedError


class ResultsQueue(Selectable):
    class Empty(Exception):
        pass

    _rpipe = FdHolder('rpipe')
    _wpipe = FdHolder('wpipe')

    def __init__(self, select_function):
        super(ResultsQueue, self).__init__()
        self._rpipe, self._wpipe = os.pipe()
        fcntl.fcntl(self._rpipe, fcntl.F_SETFL, os.O_NONBLOCK)
        self._queue = deque()
        self._select = select_function

    def __len__(self):
        return len(self._queue)

    def wait_event(self, timeout):
        try:
            if self._select([self._rpipe], [], [], timeout)[0]:
                os.read(self._rpipe, 8192)
                return True
        except Exception as e:
            if (
                (hasattr(e, 'errno') and e.errno != errno.EINTR)
                or not e.args
                or e.args[0] != errno.EINTR  # select.error isn't EnvironmentError nor has e.errno
            ):
                raise

        return False

    def pop(self):
        try:
            return self._queue.popleft()
        except IndexError:
            raise ResultsQueue.Empty()

    def append(self, result):
        self._queue.append(result)
        os.write(self._wpipe, b'1')

    def _get_event_fd(self):
        return self._rpipe

    def _is_data_ready(self):
        return bool(self._queue)

    def __del__(self):
        try:
            del self._rpipe
        finally:
            del self._wpipe


class Poll(object):
    def __init__(self, iterable=None, select_function=poll_select):
        super(Poll, self).__init__()
        self.__events = {}
        self.__select = select_function

        if iterable is not None:
            for e in iterable:
                self.add(e)

    def add(self, obj):
        assert hasattr(obj, "_get_event_fd") and hasattr(obj, "_is_data_ready"), "The object of type %s cannot be polled!" % (type(obj),)

        fd = obj._get_event_fd()
        fcntl.fcntl(fd, fcntl.F_SETFL, os.O_NONBLOCK)  # NOTE that fd mustn't be used in block-required places outside
        if fd not in self.__events:
            self.__events[fd] = obj

    def remove(self, obj):
        self.__events.pop(obj._get_event_fd(), None)

    def poll(self, timeout=None):
        start = monotime()
        fds = []
        results = []
        for fd, obj in list(self.__events.items()):
            if obj._is_data_ready():
                results.append(obj)
            else:
                fds.append(fd)

        first_time = True  # if timeout == 0 we'll poll at least one time
        while not results and (timeout is None or timeout > 0 or first_time):
            first_time = False
            try:
                ready = self.__select(fds, [], [], timeout)[0]
            except Exception as e:
                if (
                    (hasattr(e, 'errno') and e.errno != errno.EINTR)
                    or not e.args
                    or e.args[0] != errno.EINTR  # select.error isn't EnvironmentError nor has e.errno
                ):
                    raise
                if timeout is not None:
                    timeout -= monotime() - start
                start = monotime()
                continue

            for fd in ready:
                os.read(fd, 8192)
                obj = self.__events.get(fd)
                if obj is not None and obj._is_data_ready():
                    results.append(obj)

            if timeout is not None:
                timeout -= monotime() - start
            start = monotime()

        return results
