"""
Message dispatcher for custom user in-band data
"""

__author__ = 'torkve'

import sys
import functools
from collections import defaultdict, Callable
from weakref import WeakSet

from ya.skynet.util.functional import singleton

from ..window import IncomingWindow
from ..utils import genuuid, short, log as root


class RPCDispatcher(object):
    _send = None

    def __init__(self):
        self._type_handlers = defaultdict(WeakSet)
        self._id_handlers = defaultdict(WeakSet)

    def register_type_handler(self, msgtype, rpcobj):
        """
        Register a handler, based on the 'type' field
        of the message. It will handle all the messages
        of that type.

        :type msgtype: str
        :type rpcobj: RPCObject
        """
        self._type_handlers[msgtype].add(rpcobj)

    def register_id_handler(self, rpcid, rpcobj):
        """
        Register a handler, based on the 'rpcid' field
        of the message. It will handle all the messages
        coming from the selected RPC object.

        :type rpcid: str
        :type rpcobj: RPCObject
        """
        self._id_handlers[rpcid].add(rpcobj)

    def dispatch(self, addr, msg):
        msgtype = msg.get('rpctype', None)
        rpcid = msg.get('rpcid', None)
        msg = msg.get('data')

        for handler in self._type_handlers[msgtype]:
            try:
                log().debug('dispatching rpctype {} to {}'.format(msgtype, str(handler)))
                handler.process(addr[0], addr[1], msg)
            except Exception:
                log().exception('dispatching msg to {} failed:'.format(str(handler)), exc_info=sys.exc_info())
        for handler in self._id_handlers[rpcid]:
            try:
                log().debug('dispatching rpcid {} to {}'.format(short(rpcid), str(handler)))
                handler.process(addr[0], addr[1], msg)
            except Exception:
                log().exception('dispatching msg to {} failed:'.format(str(handler)), exc_info=sys.exc_info())

    def send(self, rpctype, rpcid, msg):
        self._send(('rpc', rpctype, rpcid), msg)


@singleton
def gRPCDispatcher():
    return RPCDispatcher()


class RPCObject(object):
    def __init__(self):
        super(RPCObject, self).__init__()
        self._session = None

    @property
    def needs_session(self):
        """
        If returns True, the session
        will be bound to the RPC upon serialization
        :see: :func:`.RPCObject.set_session`
        """
        return False

    def _set_session(self, session):
        self._session = session

    def process(self, hostid, addr, message):
        raise NotImplementedError


class ManyToOneMixin(object):
    def __init__(self, *args, **kwargs):
        super(ManyToOneMixin, self).__init__(*args, **kwargs)
        self._windows = defaultdict(IncomingWindow)

    def process_sequence(self, index, hostid, addr, message):
        self._windows[hostid].put(index, (hostid, addr, message))
        return self._windows[hostid].pop()


class OneToOneMixin(object):
    def __init__(self, *args, **kwargs):
        super(OneToOneMixin, self).__init__(*args, **kwargs)
        self._window = IncomingWindow()

    def process_sequence(self, index, hostid, addr, message):
        self._window.put(index, (hostid, addr, message))
        return self._window.pop()


class TrivialRPCObject(RPCObject):
    """
    Autogenerated object for RPC handling.
    :see: :func:`.handle` to mark RPC method handlers
    """

    def __init__(self, uuid=None):
        super(TrivialRPCObject, self).__init__()
        self._uuid = uuid or genuuid()
        self._handlers = self._register_handlers()

    @property
    def uuid(self):
        return self._uuid

    def register_in(self, rpc_dispatcher):
        rpc_dispatcher.register_id_handler(self._uuid, self)

    def _set_session(self, session):
        super(TrivialRPCObject, self)._set_session(session)
        self.register_in(session.rpc_dispatcher)

    def _register_handlers(self):
        h = {}

        for attr in dir(self):
            method = getattr(self, attr)
            if not isinstance(method, Callable) or not hasattr(method, '_handles'):
                continue

            h[getattr(method, '_handles')] = method.__func__

        return h

    def process_sequence(self, index, hostid, addr, message):
        raise NotImplementedError

    def process(self, hostid, addr, message):
        try:
            msgtype = message['method']
            index = message['index']
        except KeyError:
            log().warning('rpc-{} not an rpc call got from ({}) {}'.format(
                short(self._uuid), hostid, addr
            ))
            return

        for _, (hostid, addr, message) in self.process_sequence(index, hostid, addr, message):
            handler = self._handlers.get(msgtype)
            if not handler:
                log().warning('rpc-{} unknown rpc method: {}'.format(short(self._uuid), msgtype))

            kwargs = {}
            if handler._with_hostid:
                kwargs['hostid'] = hostid
            if handler._with_addr:
                kwargs['addr'] = addr
            if handler._with_host:
                kwargs['host'] = self._session.get_host_by_id(hostid)

            try:
                handler(self, message, **kwargs)
            except Exception:
                log().exception("rpc-{} failed to handle request:".format(short(self._uuid)), exc_info=sys.exc_info())


class TrivialSingleRPCObject(OneToOneMixin, TrivialRPCObject):
    pass


class TrivialMultiRPCObject(ManyToOneMixin, TrivialRPCObject):
    pass


def handle(name, with_hostid=False, with_host=False, with_addr=False):
    def wrap(fn):
        @functools.wraps(fn)
        def wrapped(*args, **kwargs):
            return fn(*args, **kwargs)
        wrapped._handles = name
        wrapped._with_hostid = with_hostid
        wrapped._with_host = with_host
        wrapped._with_addr = with_addr

        return wrapped
    return wrap


@singleton
def log():
    return root().getChild('rpc')
