"""Pipe, allowing client and servers to pull/push data"""

from ..utils import poll_select, singleton, monotime, short, log as root
from ..exceptions import CQueueRuntimeError
from ..poll import Selectable, ResultsQueue
from . import TrivialSingleRPCObject, TrivialMultiRPCObject, gRPCDispatcher, handle

import six
from collections import defaultdict

from ya.skynet.util.pickle import dumps, loads


class Empty(Exception):
    pass


class RemotePipe(TrivialSingleRPCObject):
    def __init__(self, uuid, taskid):
        super(RemotePipe, self).__init__(uuid)

        self._taskid = taskid
        self._queue = ResultsQueue(poll_select)
        self.register_in(gRPCDispatcher())
        self._index = 0

    def get(self, block=True, timeout=None):
        try:
            while not self._queue and block and (timeout is None or timeout > 0):
                t = monotime()
                self._queue.wait_event(timeout)
                if timeout is not None:
                    timeout -= monotime() - t
            return loads(self._queue.pop())
        except self._queue.Empty:
            raise Empty()

    def put(self, data):
        data = dumps(data)
        msg = push_msg(self._next_index(), data)
        gRPCDispatcher().send('pipe', self._uuid, msg)

    @handle('push')
    def _push(self, message):
        self._queue.append(message['data'])

    def _next_index(self):
        idx, self._index = self._index, self._index + 1
        return idx

    def __repr__(self):
        return '<{}>'.format(self)

    def __str__(self):
        return '{} [uuid {}]'.format(self.__class__.__name__, short(self.uuid))


class Pipe(TrivialMultiRPCObject, Selectable):
    def __init__(self, results_queue):
        super(Pipe, self).__init__()
        self._queue = results_queue
        self._indexes = defaultdict(int)

    @property
    def needs_session(self):
        return True

    def get(self, block=True, timeout=None):
        try:
            while (timeout is None or timeout > 0) and not self._queue and block:
                t = monotime()
                if self._queue.wait_event(timeout):
                    break
                if timeout is not None:
                    timeout -= monotime() - t
            src, res = self._queue.pop()
            return src, loads(res)
        except self._queue.Empty:
            raise Empty()

    def put(self, hosts, data):
        self._check_session()

        data = dumps(data)

        if isinstance(hosts, six.string_types):
            hosts = [hosts]

        for host in hosts:
            msg = push_msg(self._next_index(host), data)
            # FIXME use tree_data with msg_indexes
            self._session.rpc_send('pipe', self._uuid, msg, [host])

    def _get_event_fd(self):
        return self._queue._get_event_fd()

    def _is_data_ready(self):
        return bool(self._queue)

    def _check_session(self):
        if self._session is None:
            raise CQueueRuntimeError("Attempt to use pipe before session instantiation")

    def _next_index(self, host):
        idx, self._indexes[host] = self._indexes[host], self._indexes[host] + 1
        return idx

    @handle('push', with_host=True)
    def _push(self, msg, host):
        self._queue.append((host, msg['data']))

    def __reduce__(self):
        return RemotePipe, (self.uuid, self._session.taskid)

    def __repr__(self):
        return '<{}>'.format(self)

    def __str__(self):
        return '{} [uuid {}]'.format(self.__class__.__name__, short(self.uuid))


def push_msg(index, data):
    return {
        'index': index,
        'method': 'push',
        'data': data,
    }


@singleton
def log():
    return root().getChild('rpc.pipe')
