"""
Sign manager, that sends all the requests
to the sign manager of the client started
the session.

"""

import six

from ya.skynet.library.auth.base import BaseAuthManager
from ya.skynet.library.auth.key import CryptoKey

from . import gRPCDispatcher, RPCObject
from ..utils import genuuid, AsyncResult, log as root


class RemoteSessionAuthManager(BaseAuthManager, RPCObject):
    message_type = 'sshagent'

    def __init__(self, log=None):
        super(RemoteSessionAuthManager, self).__init__(log=log or root().getChild('rpc.sessionauth'))

        gRPCDispatcher().register_type_handler(self.message_type, self)

        self.__currentMessageId = 0
        self._requests = {}
        self.__loaded = False
        self.__data_index = 0
        self.uuid = genuuid()

    def fingerprints(self):
        self._loadIfNeeded()
        return super(RemoteSessionAuthManager, self).fingerprints()

    def hexFingerprints(self):
        self._loadIfNeeded()
        return super(RemoteSessionAuthManager, self).hexFingerprints()

    def __iter__(self):
        self._loadIfNeeded()
        return super(RemoteSessionAuthManager, self).__iter__()

    def addKey(self, key):
        raise NotImplementedError

    def removeKey(self, key):
        raise NotImplementedError

    def load(self, username=None):
        super(RemoteSessionAuthManager, self).load(username)
        msg = request_keys(self._nextMessageIndex(), self.uuid)
        response = self._sendRequest(msg)
        for key, userNames, comment in response['keys']:
            key = CryptoKey.fromNetworkRepresentation(key, log=self.log)
            key.userNames = set(userNames)
            key.comment = comment
            super(RemoteSessionAuthManager, self).addKey(key)
        self.__loaded = True

    def _loadIfNeeded(self):
        """Load keys only in case if they are not loaded yet."""
        if self.__loaded:
            return
        self.load()

    def _sendRequest(self, msg):
        result = AsyncResult()
        self._requests[msg['index']] = result
        gRPCDispatcher().send(self.message_type, self.uuid, msg)
        response = result.get()
        return response

    def process(self, hostid, addr, msg):
        request = self._requests.pop(msg['index'])
        if request:
            request.set(msg)

    def _nextMessageIndex(self):
        idx, self.__data_index = self.__data_index, self.__data_index + 1
        return idx


class RemoteSessionSignManager(RemoteSessionAuthManager):
    def sign(self, hash, fingerprints=None):
        """
        :param string hash: hash received with hashFunc().digest()
        :param fingerprints: iterable of interested fingerprints or None if all needed
        :return: iterable of (fingerprint, sign) pairs
        """
        self._loadIfNeeded()

        if isinstance(fingerprints, six.string_types):
            fingerprints = (fingerprints,)
        if not fingerprints:
            fingerprints = None

        msg = request_signs(self._nextMessageIndex(), self.uuid, hash, fingerprints)
        result = self._sendRequest(msg)['signatures']
        result = [tuple(x) for x in result]
        return result


def request_keys(index, uuid):
    return {
        'type': 'request_keys',
        'uuid': uuid,
        'index': index,
    }


def request_signs(index, uuid, hash, fingerprints):
    return {
        'type': 'request_signs',
        'uuid': uuid,
        'index': index,
        'hash': hash,
        'fingerprints': list(fingerprints),
    }
