from __future__ import print_function
import sys
import os

import six


def _chmod(path, perms):
    try:
        os.chmod(path, perms)
    except EnvironmentError:  # FreeBSD can deny sticky bit for nonroot
        os.chmod(path, perms & 0o777)


def _mkdir(dirname):
    import os
    import errno
    try:
        os.makedirs(dirname)
    except EnvironmentError as e:
        if e.errno != errno.EEXIST:
            raise


def _mount_tmpfs():
    import os
    import sys
    import pwd
    from ya.skynet.library.config import query

    try:
        config = query('skynet.services.cqudp', 'config')['server']
        username = os.getenv('SKYNET_CQUDP_USER', config['user'])
        user = pwd.getpwnam(username).pw_uid
    except (OSError, KeyError) as e:
        print(e, file=sys.stderr)
        return
    else:
        for path, size in config['Tmpfs']:
            path = path.format(userName=username)
            try:
                if not os.path.exists(path):
                    os.makedirs(path, mode=0o755)
            except EnvironmentError as e:
                print(e, file=sys.stderr)
                continue
            if os.path.ismount(path):
                continue
            if isinstance(size, six.string_types):
                size = size.lower()
                # FreeBSD knows nothing about size suffixes, so we're to convert 'em ourselves
                if size.endswith('k'):
                    size = int(size[:-1]) * 1024
                elif size.endswith('m'):
                    size = int(size[:-1]) * (1024 ** 2)
                elif size.endswith('g'):
                    size = int(size[:-1]) * (1024 ** 3)
                else:
                    print("Don't know how to mount {} ({})".format(path, size), file=sys.stderr)
                    continue
            elif not isinstance(size, six.integer_types):
                print("Size {} for path {} has invalid format".format(size, path), file=sys.stderr)
                continue
            if os.system('mount -t tmpfs -o size={},mode=01755,uid={} tmpfs {}'.format(size, user, path)) != 0:
                os.chown(path, os.getuid(), -1)
                _chmod(path, 0o1755)


def make_dir(user, dirname, perms=None):
    import os
    import pwd
    _mkdir(dirname)

    u = pwd.getpwnam(user)
    os.chown(dirname, u.pw_uid, u.pw_gid)
    if perms is not None:
        _chmod(dirname, perms)


def prepare_temp():
    from ya.skynet.library.config import query
    from ya.skynet.util.sys.user import getUserName

    config = query('skynet.services.cqudp', 'config')['server']

    tempdir = config['TempDir']

    user = os.getenv('SKYNET_CQUDP_USER', config['user'])
    if os.getuid() != 0:
        user = getUserName()

    tempdir = os.path.join(tempdir.format(userName=user), 'cqudp')

    if sys.platform != 'cygwin':
        try:
            _mount_tmpfs()
        except (EnvironmentError) as e:
            print(e, file=sys.stderr)

    make_dir('root', tempdir, 0o1755)
    # Dir for state
    make_dir('root', os.path.join(tempdir, 'state'), 0o755)
    # Dir for client temporary files
    clientTempDir = os.path.join(tempdir, 'client')
    make_dir('root', clientTempDir, 0o1777)
    tasksTempDir = os.path.join(tempdir, 'tasks')
    make_dir('root', tasksTempDir, 0o1777)

    return user, tempdir


def arc_main(args):
    user, tempdir = prepare_temp()

    environ = {}
    environ['TMPDIR'] = tempdir

    # Export all SKYNET* environment
    for key, value in os.environ.items():
        if key.startswith('SKYNET') or key in ('PYTHONPATH', 'YP_TOKEN'):
            environ[key] = value

    os.execve(
        sys.executable,
        [
            sys.executable,
            'server',
            '-u', user,
            '-t', tempdir,
            '--watch-ppid',
            '--rc-socket', 'rc-cqudp',
        ] + args.argv,
        environ
    )


def main():
    user, tempdir = prepare_temp()

    environ = {}

    # Export proper PYTHONPATH
    pythonPath = os.environ.get('PYTHONPATH', '').split(':')
    pythonPath.insert(0, os.path.abspath('lib'))
    environ['PYTHONPATH'] = ':'.join(pythonPath)
    environ['TMPDIR'] = tempdir

    # Export all SKYNET* environment
    for key, value in os.environ.items():
        if key.startswith('SKYNET'):
            environ[key] = value

    os.execve(
        sys.executable,
        [
            sys.executable,
            os.path.abspath('bin/cqudp'),
            'server',
            '-u', user,
            '-t', tempdir,
            '--watch-ppid',
            '--rc-socket', 'rc-cqudp',
        ] + sys.argv[1:],
        environ
    )

    # TODO start thinclient
