from __future__ import absolute_import

import six

from ya.skynet.util.functional import CachedState
from ya.skynet.library.auth.verify import VerifyManager, FileKeysVerifyManager
from .. import cfg

try:
    import syslog
except ImportError:
    syslog = None


def create_auth(ca_storage=None, log=None):
    fkvm = FileKeysVerifyManager(
        commonKeyDirs=cfg.server.Auth.CommonKeyDirs,
        userKeyDirs=cfg.server.Auth.UserKeyDirs,
        keyFiles=cfg.server.Auth.KeyFiles,
        caStorage=ca_storage,
        log=log,
    )

    return CachedState(fkvm, 'load', 15 * 60)


def create_empty_auth(ca_storage=None, log=None):
    vm = VerifyManager(caStorage=ca_storage, log=log)
    return vm


class AuthSyslog(object):
    DEBUG = "LOG_DEBUG"
    INFO = "LOG_INFO"
    NOTICE = "LOG_NOTICE"
    WARNING = "LOG_WARNING"
    ERROR = "LOG_ERR"
    CRITICAL = "LOG_CRITICAL"
    ALERT = "LOG_ALERT"
    EMERGENCY = "LOG_EMERG"

    def __init__(self, name):
        self.__initialized = None
        self.name = name

    def open(self):
        if self.__initialized is not None:
            return

        if syslog is None or not cfg.server.Auth.Syslog:
            self.__initialized = False
            return

        for attempt in six.moves.xrange(3):
            try:
                syslog.openlog(self.name, syslog.LOG_PID, syslog.LOG_AUTH)
                self.__initialized = True
                break
            except EnvironmentError:
                pass
        if self.__initialized is None:
            self.__initialized = False

    def __enter__(self):
        self.open()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        if self.__initialized:
            self.close()

    def log(self, level, message, *args):
        if self.__initialized:
            try:
                level = getattr(syslog, level)
            except AttributeError:
                return

            try:
                syslog.syslog(level, message % args)
            except EnvironmentError:
                pass

    def debug(self, message, *args):
        self.log(AuthSyslog.DEBUG, message, *args)

    def info(self, message, *args):
        self.log(AuthSyslog.INFO, message, *args)

    def notice(self, message, *args):
        self.log(AuthSyslog.NOTICE, message, *args)

    def warning(self, message, *args):
        self.log(AuthSyslog.WARNING, message, *args)

    def error(self, message, *args):
        self.log(AuthSyslog.ERROR, message, *args)

    def critical(self, message, *args):
        self.log(AuthSyslog.CRITICAL, message, *args)

    def alert(self, message, *args):
        self.log(AuthSyslog.ALERT, message, *args)

    def emergency(self, message, *args):
        self.log(AuthSyslog.EMERGENCY, message, *args)

    def close(self):
        if self.__initialized:
            try:
                syslog.closelog()
            except EnvironmentError:
                pass
        self.__initialized = None
