from __future__ import absolute_import

from pkg_resources import require
import hashlib
import six
import sys

from ya.skynet.util.pickle import dumps
from .manager import Manager
from ..exceptions import CQueueAuthError, CQueueRuntimeError
from ..utils import genuuid, short


class DebugManager(Manager):
    def __init__(self, privileges_lock=None, auth=None, log=None):
        super(DebugManager, self).__init__(privileges_lock=privileges_lock,
                                           auth=auth,
                                           log=log,
                                           )
        self.dowser = None
        self.ipython = None
        self.yappi = None

    def process_request(self, route, task, path, hostid, iface):
        if self.verify:
            try:
                self.verify_message(task, 'root', task['signs'], _request_hash(task), iface)
            except BaseException as e:
                ex = self.prepare_exception(e, 'Auth failure')
                if not isinstance(e, CQueueAuthError):
                    self.log.exception("[%s] error occurred during authentication: %s",
                                       short(task.get('uuid', '')),
                                       e,
                                       exc_info=sys.exc_info())
                route(response(task['uuid'], error=ex), path[:1], hostid=hostid)
                return

        try:
            service = task['service']
            action = task['action']
            method = '%s_%s' % (action, service)
            if not action or not action[0].isalpha() or not hasattr(self, method):
                self.log.info('unknown service %r and/or action %r', service, action)
                route(
                    response(task['uuid'],
                             error=CQueueRuntimeError('unknown service %r and/or action %r' % (service, action))),
                    path[:1],
                    hostid=hostid)
                return

            self.log.debug("calling %s", getattr(self, method))
            return getattr(self, method)(route, task, path, hostid, iface)
        except BaseException as e:
            self.log.exception("[%s] exception occurred while processing debug_request",
                               short(task.get('uuid', '')),
                               exc_info=sys.exc_info())
            route(
                response(task['uuid'],
                         error=CQueueRuntimeError("error while processing request: %s" % (e,))),
                path[:1],
                hostid=hostid)

    def start_pyrasite(self, route, task, path, hostid, iface):
        try:
            start_pyrasite(task['acc_host'], task['port'])
        except BaseException as e:
            ex = self.prepare_exception(e, "pyrasite start failed")
            self.log.exception("[%s] pyrasite server start failed",
                               short(task.get('uuid', '')),
                               exc_info=sys.exc_info())
            route(response(task['uuid'], error=ex), path[:1], hostid=hostid)
        else:
            route(response(task['uuid'], result='started'), path[:1], hostid=hostid)

    def start_dowser(self, route, task, path, hostid, iface):
        if self.dowser is not None:
            self.log.info('dowser is already running')
            route(
                response(task['uuid'],
                         result='dowser is already running',
                         port=self.dowser
                         ),
                path[:1],
                hostid=hostid)
            return self.dowser

        try:
            start_dowser(task['port'])
        except:
            self.log.exception("couldn't start cherrypy:", exc_info=sys.exc_info())
            stop_dowser()
            self.log.info("cherrypy stopped")
            raise

        self.log.info('cherrypy started on port %s', task['port'])
        self.dowser = task['port']
        route(response(task['uuid'], result='started', port=self.dowser), path[:1], hostid=hostid)
        return self.dowser

    def stop_dowser(self, route, task, path, hostid, iface):
        if self.dowser is None:
            route(response(task['uuid'], result='dowser is already stopped'), path[:1], hostid=hostid)
            return self.dowser

        stop_dowser()
        self.log.info("cherrypy stopped")

        self.dowser = None
        route(response(task['uuid'], result='dowser stopped'), path[:1], hostid=hostid)

    def start_yappi(self, route, task, path, hostid, iface):
        if self.yappi is not None:
            self.log.info("yappi is already running")
            route(
                response(task['uuid'],
                         error=CQueueRuntimeError('yappi is already running'),
                         ),
                path[:1],
                hostid=hostid)
            return self.yappi

        try:
            start_yappi()
        except:
            self.log.exception("couldn't start yappi:")
            raise

        self.log.info('yappi started (results will be written to %s)', task['port'])
        self.yappi = task['port']
        route(response(task['uuid'], result='started', port=self.yappi), path[:1], hostid=hostid)
        return self.yappi

    def stop_yappi(self, route, task, path, hostid, iface):
        if self.yappi is None:
            route(response(task['uuid'], error=CQueueRuntimeError('yappi is already stopped')), path[:1], hostid=hostid)
            return self.yappi

        stop_yappi(self.yappi)
        self.yappi = None
        route(response(task['uuid'], result='yappi stopped'), path[:1], hostid=hostid)


def start_yappi():
    import yappi
    yappi.start(True, True)


def stop_yappi(path):
    import yappi
    yappi.stop()
    stats = yappi.get_func_stats()
    stats.save(path, 'callgrind')
    yappi.clear_stats()


def start_dowser(port):
    require('pil')
    import cherrypy
    import dowser

    cherrypy.tree.mount(dowser.Root())
    cherrypy.config.update({
        'server.socket_host': '::',
        'server.socket_port': port,
        'environment': 'embedded',
    })
    cherrypy.server.start()
    cherrypy.engine.start()


def stop_dowser():
    import cherrypy
    cherrypy.engine.stop()
    cherrypy.server.stop()


def start_pyrasite(host, port):
    from ..debugtools.reverse import ReversePythonShell
    ReversePythonShell(host, port).start()


def response(request, result=None, error=None, port=None):
    return {
        'uuid': genuuid(),
        'type': 'debug_response',
        'request': request,
        'result': result,
        'port': port,
        'error': dumps(error) if error is not None else None,
    }


def _request_hash(task):
    md5 = hashlib.md5(six.b(task['uuid']))
    md5.update(six.b(task['acc_user']))
    md5.update(six.b(task['acc_host']))
    md5.update(six.b(task['service']))
    md5.update(six.b(task['action']))
    md5.update(six.b(str(task['port'])))
    md5.update(six.b(str(task['ctime'])))
    return md5.digest()
