import os
import sys
import shutil
import marshal

from ...utils import singleton, log as root
from ... import msgpackutils as msgpack

from ya.skynet.util import pickle

import six


def _arcadia_binary_child(task, on_ready):
    data = msgpack.loads(task['data'])
    pickled = data['object']
    custom_main_pyc = data.get('exec_custom_main')
    pickled_arg = task['options'].get('exec_extra_params')

    if custom_main_pyc is not None:
        custom_main = marshal.loads(custom_main_pyc[8:])
        main = sys.modules['__main__']
        six.exec_(custom_main, main.__dict__)

    obj = pickle.loads(pickled)
    on_ready()

    try:
        if pickled_arg is not None:
            return obj(pickle.loads(pickled_arg))
        else:
            return obj()
    except Exception as err:
        log().exception("%s", err)
        raise


def _collect_torrent_sizes(cache_path):
    _, dirnames, filenames = next(os.walk(cache_path))

    for filename in filenames:
        os.unlink(os.path.join(cache_path, filename))

    torrents = {
        dirname: 0
        for dirname in dirnames
    }
    total = 0

    for torrent in torrents:
        for dirname, _, filenames in os.walk(os.path.join(cache_path, torrent)):
            for filename in filenames:
                size = os.stat(os.path.join(dirname, filename)).st_size
                torrents[torrent] += size
                total += size

    stats = [
        (
            os.path.join(cache_path, torrent),
            os.stat(os.path.join(cache_path, torrent)).st_mtime,
            torrents[torrent]
        )
        for torrent in torrents
    ]
    stats.sort(key=lambda item: item[1])
    return total, stats


def _cleanup_cache(cache_path, required_to_download=0, skip_path=None):
    cache_limit = 5 * 1024 * 1024 * 1024  # 5 GB
    total, stats = _collect_torrent_sizes(cache_path)

    while stats and total + required_to_download > cache_limit:
        path, _, size = stats.pop()
        if path == skip_path:
            continue

        log().info("will remove %r (size=%s) from cache", path, size)
        shutil.rmtree(path)
        total -= size


def _download_binary(rbtorrent, torrent_size=None):
    cache_path = os.path.expanduser('~/.local/cache/cqudp/')
    destination_path = os.path.join(cache_path, rbtorrent)

    if not os.path.exists(cache_path):
        os.makedirs(cache_path)

    import api.copier
    copier = api.copier.Copier()
    handle = copier.handle(rbtorrent)

    # required_to_download = sum(
    #     item['size']
    #     for item in files
    #     if item.get('type') == 'file'
    # )

    if torrent_size:
        _cleanup_cache(cache_path, torrent_size)

    log().info("will download %r", rbtorrent)
    try:
        result = handle.get(dest=destination_path, subproc=True).wait()
    except api.copier.FilesystemError as e:
        # all kudos should be sent to mocksoul
        if "No space left" not in str(e):
            raise

        # just clean everything
        _cleanup_cache(cache_path, sys.maxsize)
        result = handle.get(dest=destination_path, subproc=True).wait()

    # remove excessive data after download
    _cleanup_cache(cache_path, 0, destination_path)

    # take first executable from torrent
    binary = next(
        iter(
            filter(
                lambda record: record.get("executable") and record.get("type") == "file",
                result.files()
            )
        ),
        None
    )
    if not binary:
        raise RuntimeError("torrent does not contain any executable binary")

    return binary['path']


def arcadia_binary(task, on_ready):
    if os.getenv('CQUDP_IN_BINARY'):
        return _arcadia_binary_child(task, on_ready)

    data = msgpack.loads(task['data'])
    rbtorrent = data['resource_id']
    entry_point = data['entry_point']
    torrent_size = data.get('size')

    bin_path = _download_binary(rbtorrent, torrent_size)
    os.environ['Y_PYTHON_ENTRY_POINT'] = str(entry_point)
    os.environ['CQUDP_IN_BINARY'] = 'yes'
    log().info("calling exec(%r)", bin_path)
    os.execv(bin_path, [bin_path])


@singleton
def log():
    return root().getChild('exec_functions.arc')
