import os
import errno
import fcntl
import textwrap

from .common import STDOUT, STDERR, start_cmd
from ...utils import poll_select


OVERFLOW_ERROR = textwrap.dedent(
    '''
    cqudp error: process output exceeds %d bytes limit: %d bytes, output has been wiped.
    cqudp error: Do not send that many data with cqudp,
    cqudp error: it can eat the backbone network and admins will be evil with you.
    cqudp error: Hint: for large output use skynet copier.
    '''
)

STREAM_OVERFLOW_ERROR = textwrap.dedent(
    '''
    cqudp error: process output exceeds %d bytes limit: %d bytes, remaining output has been wiped.
    cqudp error: and process has been terminated.
    cqudp error: Do not send that many data with cqudp,
    cqudp error: it can eat the backbone network and admins will be evil with you.
    cqudp error: Hint: for large output use skynet copier.
    '''
)


def shell_task(task, on_ready):
    on_ready()
    cmd = task['options']['exec_args']['cmd']
    stream = task['options']['exec_args'].get('stream', False)
    data_limit = task['options'].get('data_limit', None) or None
    data_limit = max(data_limit, 16384) if data_limit is not None else None  # to be able to send errors
    env = task['options'].get('extra_env')
    in_rtc_container = task['options'].get('in_rtc_container', False)

    if not stream:
        process = start_cmd(cmd, extra_env=env, in_rtc_container=in_rtc_container)
        out = process.communicate()
        total = len(out[0]) + len(out[1])
        if data_limit is not None and total > data_limit:
            err = OVERFLOW_ERROR % (data_limit, total)
            yield '', err, process.returncode
        else:
            yield out[0], out[1], process.returncode
    else:
        process = start_cmd(cmd, extra_env=env, in_rtc_container=in_rtc_container)
        ios = [process.stdout.fileno(), process.stderr.fileno()]

        for fd in ios:
            flags = fcntl.fcntl(fd, fcntl.F_GETFL)
            fcntl.fcntl(fd, fcntl.F_SETFL, flags | os.O_NONBLOCK)

        total = 0
        for res in _collect_process_output(process, ios):
            total += res[1]
            if data_limit is not None and total > data_limit:
                err = STREAM_OVERFLOW_ERROR % (data_limit, total)
                yield (STDERR, err)
                process.stdout.close()
                process.stderr.close()
                process.terminate()
                return

            yield res


def _collect_process_output(process, ios):
    collected = [[], []]

    while ios:
        ready = poll_select(ios, [], [])[0]

        for data, mark, stream in (
            (collected[0], STDOUT, process.stdout),
            (collected[1], STDERR, process.stderr)
        ):
            if stream.fileno() not in ready:
                continue

            try:
                piece = stream.read()
            except EnvironmentError as err:
                if err[0] == errno.EBADF:
                    return
                elif err[0] == errno.EAGAIN:
                    continue
                else:
                    raise

            if not piece:
                if data:
                    yield (mark, ''.join(data))
                    del data[:]
                ios.remove(stream.fileno())
                continue
            elif '\n' in piece:
                piece = (''.join(data) + piece).split('\n')
                del data[:]
                for x in piece[:-1]:
                    yield(mark, x + '\n')
                if piece[-1]:
                    data.append(piece[-1])
            else:
                data.append(piece)
