from ...exceptions import CQueueExecutionFailure
from ...utils import log as root

import six

from ya.skynet.util.errors import saveTraceback, getTraceback, setTraceback


class Job(object):
    def __init__(self, taskid):
        self.taskid = taskid

    def terminate(self):
        raise NotImplementedError

    def _wait_process(self):
        """
        Wait until process finish

        :return: int process returncode
        """
        raise NotImplementedError

    def _process_returncode(self, retcode):
        if retcode > 0:
            # * TODO we SHOULD pop uuid in case of retriable tasks and notify client
            # * about it so that it could send task again and ignore exception
            # self.tasks.pop(uuid, None)
            raise CQueueExecutionFailure(
                "Task {} spawn failed (errorcode {}). See host cqudp log for details".format(self.taskid, retcode)
            )
        elif retcode and retcode < 0:
            # currently there's no way to find proper message index
            # to send message if the parent bus was killed,
            # so we just ignore it, hoping that no one would kill the bus
            pass

    def join(self):
        """
        :raises CQueueExecutionFailure: if the task is terminated abnormally
        """
        try:
            retcode = self._wait_process()
            self._process_returncode(retcode)
        except Exception as e:
            saveTraceback(e)
            newExc = CQueueExecutionFailure(str(e))
            setTraceback(newExc, getTraceback(e))
            raise newExc

    def get_cpu(self):
        return 0

    def get_memory(self):
        return 0


class ExecuterMetaclass(type):
    def __new__(mcs, name, bases, cls_dict):
        new_type = type.__new__(mcs, name, bases, cls_dict)
        if name == 'Executer':
            return new_type

        name = cls_dict.get('name')
        if name is not None:
            # prev_type = Executer._known_.get(name)
            # assert prev_type is None, 'executer name collision in %s and %s' % (name, str(prev_type))
            Executer._known_[name] = new_type

        return new_type


@six.add_metaclass(ExecuterMetaclass)
class Executer(object):
    _known_ = {}
    name = None

    def __init__(self, lock, log=None):
        self.lock = lock
        self.log = log or root().getChild('executer')

    def execute(self, uuid, args, user, home, cgroups_path=None, cgroup=None, extra_env=None):
        """
        :param str uuid: task uuid
        :param list args: shell args list
        :param str user: user to execute task as
        :param str home: dir to chdir into
        :param dict extra_env: additional environment params
        :return: :class:`~Job`
        """
        raise NotImplementedError

    @classmethod
    def _available(cls):
        """
        Check if the executer can be used

        :return: bool
        """
        raise NotImplementedError

    @classmethod
    def _resolve(cls, name):
        return cls._known_.get(name)

    @classmethod
    def available(cls, name):
        klass = cls._resolve(name)
        if klass is None or not klass._available():
            return False
        return True

    @classmethod
    def create(cls, name, lock, log=None):
        klass = cls._resolve(name)
        assert klass is not None, 'Unknown executer type'
        return klass(lock, log)

    @property
    def name(self):
        raise NotImplementedError
