from __future__ import absolute_import
import os
import pwd
import grp
import sys

from ...mocksoul_rpc.socket import Socket
from ... import containers_cfg
from ...utils import short, sleep
from ..rootwrapper import get_portoconn
from .base import Job, Executer

import six

try:
    from infra.skylib import porto as portotools
    from infra.skylib import safe_container_actions

    from porto.api import rpc as rpc_pb2
except ImportError:
    portotools = None  # hence, it's not server, so just forget it
    safe_container_actions = None
    rpc_pb2 = None

from porto import exceptions
from porto import Connection


class PortoJob(Job):
    def __init__(self, rpc, taskid, contname):
        super(PortoJob, self).__init__(taskid)
        self.rpc = rpc
        self.contname = contname
        self.used_cpu = 0
        self.used_mem = 0

    def terminate(self):
        try:
            self.rpc.Stop(self.contname)
        except (exceptions.ContainerDoesNotExist, exceptions.InvalidState):
            pass

    def get_cpu(self):
        return self.used_cpu

    def get_memory(self):
        return self.used_mem

    def _wait_process(self):
        while self.rpc.GetData(self.contname, 'state') == 'running':
            sleep(1)

        try:
            try:
                retcode = int(self.rpc.GetData(self.contname, 'exit_status'))

                self.used_cpu = float(self.rpc.GetData(self.contname, 'cpu_usage')) / 1e9
                self.used_mem = self.rpc.GetData(self.contname, 'max_rss')
            except Exception:  # container was stopped, what we have to return here?
                return 0

            if os.WIFSIGNALED(retcode):
                return -os.WTERMSIG(retcode)
            else:
                return os.WEXITSTATUS(retcode)
        finally:
            self.rpc.Destroy(self.contname)


class PortoExecuter(Executer):
    name = 'porto'

    def __init__(self, lock, log=None):
        super(PortoExecuter, self).__init__(lock, log)

        self.rpc = get_portoconn(lock)

    @staticmethod
    def _quote_str(s):
        io = six.moves.cStringIO()
        io.write("'")
        for char in s:
            if char == "'":
                io.write("'\"'\"'")
            else:
                io.write(char)
        io.write("'")
        return io.getvalue()

    @staticmethod
    def _quote_env(env):
        return ';'.join(
            '{}={}'.format(k, v.replace(';', r'\;'))
            for (k, v) in six.iteritems(env)
            if k in ('SHELL', 'TERM', 'PATH', 'PYTHONPATH',)
        )

    @staticmethod
    def _build_command(args):
        first = True
        io = six.moves.cStringIO()

        for arg in args:
            if not first:
                io.write(' ')
            io.write(PortoExecuter._quote_str(arg))
            first = False

        return io.getvalue()

    def map_to_host(self, container, path):
        return self.rpc.ConvertPath(path, container, "self")

    def map_to_container(self, container, path):
        return self.rpc.ConvertPath(path, "self", container)

    def exec_in_container(self, container, fn, *args, **kwargs):
        def job(sock):
            try:
                result = fn(*args, **kwargs)
            except EnvironmentError as e:
                sock.sendall(b'\x02' + bytes(str(e.errno)))
            except Exception as e:
                sock.sendall(b'\x01' + bytes(str(e)))
            else:
                sock.sendall(b'\x00')
                s = Socket(sock)
                s.write_msgpack(result)
            finally:
                sock.close()

        root_pid = int(self.rpc.GetData(container, 'root_pid'))

        sock, pid = safe_container_actions.run_process(job, target_user=None, root_pid=root_pid)
        try:
            result = sock.recv(1)
            if result == b'\x01':
                raise Exception(sock.recv(4096))
            elif result == b'\x02':
                err = int(sock.recv(4))
                raise EnvironmentError(err, os.strerror(err))
            else:
                s = Socket(sock)
                return next(s.read_msgpack())
        finally:
            os.kill(pid, 9)
            sock.close()

    def get_container_user(self, container, user=None):
        return portotools.get_container_user_and_group(self.rpc, container, user)

    def _create_container(self, name):
        job_spec = rpc_pb2.TContainerSpec()
        job_spec.name = name
        job_spec.owner_containers = "self"

        return self.rpc.CreateSpec(job_spec, start=False)

    def ensure_root_container(self, user):
        root_container_name = containers_cfg.users.get(user, containers_cfg.users['ALL'])
        parts = root_container_name.split('/')
        cfg = containers_cfg.data

        for idx, part in enumerate(parts):
            name = '/'.join(parts[:idx + 1])
            cfg = cfg['containers'][part]

            try:
                container = self._create_container(name)
            except exceptions.ContainerAlreadyExists:
                container = self.rpc.Find(name)
                state = container.GetProperty('state')
                if state == 'meta':
                    continue
                elif state != 'stopped':
                    self.rpc.Destroy(name)
                    container = self._create_container(name)

            try:
                opts = dict(
                    isolate=False,
                    respawn=False,
                    virt_mode='host',
                    private='CQUDP',
                )
                opts.update(cfg['options'])
                for opt, value in six.iteritems(opts):
                    self.log.info("[{}] Setting {} {!r}".format(name, opt, value))
                    container.SetProperty(opt, value)
                container.Start()
            except exceptions.InvalidState:
                exc_info = sys.exc_info()
                state = container.GetProperty('state')
                if state != 'meta':
                    six.reraise(*exc_info)

        return container

    def execute(self,
                uuid,
                args,
                user,
                home,
                cgroups_path=None,
                cgroup=None,
                extra_env=None,
                container=None,
                **porto_args
                ):
        # during sleep time porto could have been silently disconnected
        # because of many reasons, so we just do an empty call to reconnect
        # if required
        try:
            self.rpc.Version()
        except exceptions.SocketError:
            pass

        taskid = short(uuid)
        contname = "cqudp-{}".format(taskid)
        if container is None:
            container = self.ensure_root_container(user)
            virt_mode = 'host'
        else:
            virt_mode = 'app'

        cmd = self._build_command(args)
        env = os.environ.copy()
        env.pop('YP_TOKEN', None)
        env['PYTHONDONTWRITEBYTECODE'] = '1'
        env['PYTHONNOUSERSITE'] = '1'
        env.update(extra_env or {})
        env = self._quote_env(env)

        gid = pwd.getpwnam(user).pw_gid
        group = grp.getgrgid(gid).gr_name

        porto_args.update(
            command=cmd,
            isolate=False,
            respawn=False,
            private='CQUDP-TASK',
            user=user,
            group=group,
            env=env,
            cwd=home,
            virt_mode=virt_mode,
        )

        parent_name = container if isinstance(container, six.string_types) else container.name
        contname = parent_name + '/' + contname

        job = self._create_container(contname)

        # FIXME unify with portoshell
        own_user = job.GetProperty('owner_user')
        if own_user == 'root':
            if virt_mode == 'host':
                porto_args.update(dict(
                    owner_user=user,
                    owner_group=group,
                ))
            else:
                parent = self.rpc.Find(parent_name)
                porto_args.update(dict(
                    owner_user=parent.GetProperty('owner_user'),
                    owner_group=parent.GetProperty('owner_group'),
                ))

        # job_spec = rpc_pb2.TContainerSpec()
        # job_spec.name = contname
        # job_spec.private = 'CQUDP-TASK'
        # job_spec.command = cmd
        # for k, v in six.iteritems(env):
        #     env_var = job_spec.env.var.add()
        #     env_var.name = k
        #     if v is None:
        #         env_var.unset = True
        #     else:
        #         env_var.value = v

        for argname, argval in six.iteritems(porto_args):
            self.log.info("[%s] Setting %s %r", taskid, argname, argval)
            job.SetProperty(argname, argval)

        if virt_mode != 'host':
            try:
                argname = 'enable_porto'
                argval = 'child-only'
                self.log.info("[%s] Setting %s %r", taskid, argname, argval)
                job.SetProperty(argname, argval)
            except (exceptions.InvalidValue, exceptions.PermissionError):
                argval = False
                self.log.info("[%s] Setting %s %r", taskid, argname, argval)
                job.SetProperty(argname, argval)

            portotools.set_capabilities(self.rpc, job, user, own_user)

        try:
            job.Start()
        except Exception:
            exc_info = sys.exc_info()
            try:
                job.Destroy()
            except:
                pass
            six.reraise(*exc_info)

        return PortoJob(self.rpc, uuid, contname)

    @classmethod
    def _available(cls):
        try:
            Connection(timeout=20).TryConnect()
        except Exception:
            return False
        else:
            return True
