import os

from ya.skynet.util import logging

from ...utils import short, as_user, sleep
from ..rootwrapper import RootWrapper, has_root
from .base import Job, Executer


class ProcmanJob(Job):
    def __init__(self, user, process, taskid, has_root, lock, log):
        super(ProcmanJob, self).__init__(taskid)
        self.user = user
        self.process = RootWrapper(process, lock, has_root)
        self.used_mem = 0
        self.used_cpu = 0
        self.log = logging.MessageAdapter(
            log,
            fmt='[%(taskid)s] %(message)s',
            data={'taskid': taskid},
        )

    def terminate(self):
        self.process.signal('SIGINT')

    def get_memory(self):
        return self.used_mem

    def get_cpu(self):
        return self.used_cpu

    def _wait_process(self):
        while self.process.running():
            sleep(1)
        st = self.process.stat()
        retcodes = st['retcodes']
        if retcodes:
            self.log.info("exited with '{}'".format(retcodes))
            retcode = retcodes[0]

            self.used_cpu = st.get('cpu_usage', [0])[-1]
            self.used_mem = st.get('memory_usage', [0])[-1]

            if 'signal' in retcode:
                retcode = -int(retcode.split(' ')[-1])
            else:
                retcode = int(retcode.split(' ')[-1])

            self.log.info("finished with {}".format(retcode))
            return retcode

        errors = st.get('errors')
        if errors:
            self.log.info("execution failed with '{}'".format(errors))
            raise errors[-1]

        self.log.warning("just disappeared from procman")
        return 255


class ProcmanExecuter(Executer):
    name = 'procman'

    def __init__(self, lock, log=None):
        super(ProcmanExecuter, self).__init__(lock, log)

        if has_root():
            # NOTE not evaluated under lock, because the __init__ itself is executed under lock. Ugly, but anyway
            self.executer = as_user('root', ProcmanExecuter._connect, lock, True)
        else:
            self.log.warning("no root available: connecting procman as user")
            self.executer = ProcmanExecuter._connect(lock, False)

    @classmethod
    def _connect(cls, lock=None, has_root=True):
        from api.procman import ProcMan

        executer = ProcMan()
        # executer.find_by_tags(['nonexistent_tag_kokoko'])  # enforce procman api to connect
        return RootWrapper(executer, lock, has_root)

    def execute(self,
                uuid,
                args,
                user,
                home,
                cgroups_path=None,
                cgroup=None,
                porto=True,
                porto_options=None,
                hostid=None,
                extra_env=None,
                ):
        env = {
            'PYTHONPATH': os.getenv('PYTHONPATH', ''),
            'PYTHONDONTWRITEBYTECODE': '1',
            'PYTHONNOUSERSITE': '1',
        }
        env.update(extra_env or {})

        uuid = uuid if hostid is None else '%s-%s' % (uuid, hostid)
        porto_options = porto_options or {}
        # porto_options['owner_user'] = user
        process = self.executer.create(
            args=args,
            user=user,
            cwd=home,
            tags=['cqudp-task'],
            keeprunning=False,
            uuid=uuid,
            cgroup=cgroup,
            liner=True,
            logStderr=True,
            env=env,
            porto=porto,
            porto_options=porto_options,
        )
        taskid = short(uuid)
        self.log.info("task {} spawned bus (procman uuid {})".format(taskid, uuid))
        return ProcmanJob(user, process, taskid, has_root(), self.lock, self.log)

    @classmethod
    def _available(cls):
        try:
            ProcmanExecuter._connect(False)
        except Exception:
            return False
        else:
            return True
