from __future__ import absolute_import
import os
import errno
import signal
import subprocess

from ...utils import short, as_user, UserPrivileges
from ..rootwrapper import has_root
from .base import Job, Executer


class Popen(subprocess.Popen):
    if not subprocess.mswindows:
        def wait(self):
            while self.returncode is None:
                try:
                    pid, sts, rusage = subprocess._eintr_retry_call(os.wait4, self.pid, 0)
                except OSError as e:
                    if e.errno != errno.ECHILD:
                        raise
                    pid = self.pid
                    sts = 0
                if pid == self.pid:
                    self._handle_exitstatus(sts)
                    self._handle_rusage(rusage)
            return self.returncode

        def _handle_rusage(self, rusage):
            self.rusage_cpu = rusage.ru_utime + rusage.ru_stime

        @property
        def used_cpu(self):
            return getattr(self, 'rusage_cpu', 0)


class SubprocessJob(Job):
    def __init__(self, process, taskid, user):
        super(SubprocessJob, self).__init__(taskid)
        self.process = process
        self.pid = process.pid
        self.user = user
        self.exitcode = None
        self.used_cpu = 0

    def terminate(self):
        try:
            as_user(self.user, self.process.send_signal, signal.SIGINT)
        except:
            pass

    def _wait_process(self):
        if self.exitcode is None:
            self.process.communicate()
            self.used_cpu = self.process.used_cpu
            self.exitcode = self.process.returncode
            self.process = None
        return self.exitcode

    def get_cpu(self):
        return self.used_cpu


class SubprocessExecuter(Executer):
    name = 'subprocess'

    def execute(self, uuid, args, user, home, cgroups_path=None, cgroup=None, extra_env=None):
        if cgroups_path and cgroup:
            args = args + ['--cgroups-path', cgroups_path, '--cgroup', cgroup]

        args += ['--chdir', home]

        is_root = has_root()

        env = os.environ.copy()
        env.pop('TMPDIR', None)
        env.pop('YP_TOKEN', None)
        env.update(extra_env or {})

        if is_root:
            args += ['--user', user]
            with UserPrivileges(modifyGreenlet=False):
                process = Popen(args, shell=False, close_fds=True, env=env)
        else:
            process = Popen(args, shell=False, close_fds=True, env=env)

        taskid = short(uuid)
        self.log.info("task {} spawned bus (pid {})".format(taskid, process.pid))
        return SubprocessJob(process, taskid, user)

    @classmethod
    def _available(cls):
        return True
