from __future__ import absolute_import
import os
import types
import socket
import functools
import contextlib
from ..utils import UserPrivileges

from ya.skynet.util.functional import memoized

from porto import Connection
from porto.exceptions import SocketError, SocketTimeout, Permission


@contextlib.contextmanager
def auto_user_privileges():
    if os.getuid() != 0 and has_root():
        with UserPrivileges(modifyGreenlet=False):
            yield
    else:
        yield


class RootWrapper(object):
    def __init__(self, slave, lock, as_root):
        self.__as_root = as_root
        self.__slave = slave
        self.__lock = lock

    def __getattr__(self, attr):
        if not self.__as_root:
            return getattr(self.__slave, attr)

        with self.__lock:
            with auto_user_privileges():
                return RootWrapper(getattr(self.__slave, attr), self.__lock, True)

    def __call__(self, *args, **kwargs):
        if not self.__as_root:
            return self.__slave(*args, **kwargs)

        with self.__lock:
            with auto_user_privileges():
                return self.__slave(*args, **kwargs)


def has_root():
    res = []
    for attr in ('getresuid', 'getreuid'):
        if hasattr(os, attr):
            res = getattr(os, attr)()
            return 0 in res

    for attr in ('geteuid', 'getuid'):
        if hasattr(os, attr):
            if getattr(os, attr)() == 0:
                return True

    return False


class PortoAdapter(object):
    def __init__(self, slave, conn=None):
        self.__slave = slave

    @staticmethod
    def _is_porto(obj):
        if not str(getattr(obj, '__module__', '')).startswith('porto.'):
            return False
        if not hasattr(obj, '__class__'):
            return False
        if '_RPC' == obj.__class__.__name__:
            return False
        return True

    def __wrap(self, item):
        if callable(item):
            return self.__wrapper(item)
        elif self._is_porto(item):
            return PortoAdapter(item)
        elif isinstance(item, (list, types.GeneratorType)):
            return [
                (PortoAdapter(x) if self._is_porto(x) else x)
                for x in item
            ]
        else:
            return item

    def __get_slave(self):
        slave = self.__slave
        if hasattr(slave, 'api'):
            slave = slave.api
        elif hasattr(slave, 'conn'):
            slave = slave.conn

        return slave

    def __try_connect(self):
        slave = self.__get_slave()

        if hasattr(slave, 'rpc') and slave.rpc.sock is None:
            slave.TryConnect()  # old version of api
        elif hasattr(slave, 'Connected') and not slave.Connected():
            slave.Connect()  # old version of api

    def __wrapper(self, cb):
        @functools.wraps(cb)
        def _fn(*args, **kwargs):
            for attempt in range(3):
                try:
                    self.__try_connect()
                    result = cb(*args, **kwargs)
                    return self.__wrap(result)
                except Permission:
                    # in some cases (some race maybe?) porto believes we are not root and forbids our
                    # request
                    self.__get_slave().Disconnect()
                    if attempt == 2:
                        raise
                except (socket.error, SocketError, SocketTimeout):
                    if attempt == 2:
                        raise
        return _fn

    def __getattr__(self, item):
        res = getattr(self.__slave, item)
        return self.__wrap(res)


@memoized
def get_portoconn(lock):
    return RootWrapper(PortoAdapter(Connection(timeout=20, auto_reconnect=False)), lock, has_root())
