from __future__ import absolute_import

import os
import sys
import base64

from ya.skynet.util import logging
from ya.skynet.util.console import setProcTitle
from ya.skynet.util.pickle import loads
from ya.skynet.util.sys.user import UserPrivileges

from ..utils import log as root, short, configure_log
from .. import msgpackutils as msgpack

from .taskhandle import TaskHandle


in_arcadia = bool(getattr(sys, 'is_standalone_binary', False))
arg_base = [] if in_arcadia else [sys.executable]


def set_cgroup(log, path, name):
    if name is None or path is None:
        return

    if not os.path.isdir(path):
        log.warning("cgroups path is not a directory, cgroup not installed: `%s`", path)
        return

    pid = os.getpid()

    for section in os.listdir(path):
        tasksPath = os.path.join(path, section, name, "tasks")
        if not os.path.exists(tasksPath) or not os.path.isfile(tasksPath):
            log.warning("cannot add task pid %d to cgroup %r: file does not exist", pid, name)
            continue

        try:
            open(tasksPath, 'wb').write("{}\n".format(pid))
        except Exception as err:
            log.exception("cannot add task pid %d to cgroup %r: %s",
                          pid,
                          name,
                          err
                          )

    args = arg_base + [arg for arg in sys.argv]
    cgroup_index = args.index('--cgroup')
    del args[cgroup_index:cgroup_index + 2]
    cgroup_index = args.index('--cgroups-path')
    del args[cgroup_index:cgroup_index + 2]
    log.info("calling exec w/o cgroup: %r %s", sys.executable, args)
    os.execv(sys.executable, args)


def set_user(log, user):
    if user is None:
        return

    os.setsid()
    UserPrivileges(user=user, store=False, limit=True, modifyGreenlet=False).__enter__()
    args = arg_base + [arg for arg in sys.argv]
    user_index = args.index('--user')
    del args[user_index:user_index + 2]
    log.info("calling exec w/o user: %r %s", sys.executable, args)
    os.execv(sys.executable, args)


def chdir(path):
    if path is not None:
        os.chdir(path)


def safe_remove(path):
    if os.path.isfile(path):
        try:
            os.remove(path)
        except EnvironmentError:
            pass


def main(args):
    configure_log(suffix='child', level=args.loglevel, debug=args.debug)
    log = logging.MessageAdapter(
        root().getChild('task'),
        fmt='[%(uuid)s] %(message)s',
        data={'uuid': short(args.taskid)}
    )

    set_cgroup(log, args.cgroups_path, args.cgroup)
    set_user(log, args.user)
    chdir(args.chdir)

    with open(args.task, 'rb') as task_file:
        task_val = loads(task_file.read())
    # it's a pity if something prevented us from removing file
    # (e.g. we were starting for 5+ minutes and it was already collected)
    # but removal is not required for our job
    safe_remove(args.task)

    retpath = msgpack.loads(base64.decodestring(args.retpath))
    notify_path = msgpack.loads(base64.decodestring(args.notify)) if args.notify is not None else None

    sys.stdin.close()
    sys.stdin = open('/dev/null', 'rb')
    os.dup2(sys.stdin.fileno(), 0)

    log.info('initializing cqudp task from %s', retpath[0])
    setProcTitle('cqudptsk-{}'.format(short(args.taskid)))

    try:
        t = TaskHandle(
            args.taskid,
            args.hostid,
            task_val,
            retpath,
            notify_path=notify_path,
        )
        t.run()
    finally:
        log.info("task stopped")
