from __future__ import absolute_import

import grp
import pwd
import six
import sys
import os
import copy
import base64
import errno
import threading
import tempfile
import hashlib
import time
from collections import defaultdict
from functools import wraps

from ..utils import LRUCache, genuuid, short, run_daemon, as_user, monotime, log as root
from ..exceptions import CQueueRuntimeError, CQueueAuthError
from .. import cgroups_cfg, cfg, msgpackutils as msgpack
from .rootwrapper import has_root
from .executer import Executer
from .taskhandle import result, addpath_msg
from .manager import Manager

from ya.skynet.util.sys.user import getUserName, getUserHome
from ya.skynet.util.pickle import dumps
from ya.skynet.util.errors import getTraceback
from ya.skynet.util.functional import singleton

try:
    from ya.skynet.services.portoshell.slots.slot import normalize_ip
except ImportError:
    normalize_ip = lambda x: x


def logged(fn):
    @wraps(fn)
    def wrapper(*args, **kwargs):
        try:
            return fn(*args, **kwargs)
        except Exception as e:
            log().exception('%s failed: %s', str(fn), e, exc_info=sys.exc_info())

    return wrapper


class TaskManager(Manager):
    in_arcadia = bool(getattr(sys, 'is_standalone_binary', False))
    in_rtc_container = False

    if in_arcadia or 'python' not in os.path.basename(sys.executable):
        interpreter = sys.executable
        taskpath = None
    else:
        interpreter = sys.executable
        # not stable against 'python -c ...', but we'll probably never use flags
        taskpath = sys.argv[0]

    # maximum timeout to wait for tasks spawn. Used only when shutting down the taskmgr
    tasks_start_timeout = 90.0

    def __init__(self,
                 privileges_lock=None,
                 executers=None,
                 interpreter_py2=None,
                 interpreter_py3=None,
                 auth=None,
                 log=None,
                 keys_storage=None,
                 ca_storage=None,
                 ):
        super(TaskManager, self).__init__(privileges_lock=privileges_lock,
                                          auth=auth,
                                          log=log,
                                          keys_storage=keys_storage,
                                          ca_storage=ca_storage,
                                          )
        self.has_root = has_root()
        self.tasks = LRUCache(300)
        self.stats = defaultdict(lambda: {'cpu': 0, 'memory': 0, 'count': 0})
        self.executers = []
        self.stopped_tasks = set()
        self.cgroups_mgr = _CGroupsManager()
        self.interpreter_py2 = interpreter_py2 or '/usr/bin/python2'
        self.interpreter_py3 = interpreter_py3 or '/usr/bin/python3'

        self._awaiting_jobs = set()
        self._job_ensure_lock = threading.Lock()
        self._job_spawn_lock = threading.Lock()
        self._job_spawned = threading.Condition(self._job_spawn_lock)

        if executers is None or all(isinstance(executer, six.string_types) for executer in executers):
            names = list(executers or cfg.server.Executers)
            if not names:
                raise RuntimeError("Cannot initialize TaskManager: no process executers available")
            with self.user_lock:
                self.log.info("initializing taskmgr with executers: {}".format(names))
                for executer in names:
                    if Executer.available(executer):
                        self.log.debug('initializing executer %s', executer)
                        self.executers.append(Executer.create(executer, self.user_lock, log=self.log))
        else:
            names = [executer.name for executer in executers]
            self.log.info("initializing taskmgr with executers: {}".format(names))
            self.executers = executers

        if 'porto' in names:
            self.container_executer = next(iter(filter(lambda x: x.name == 'porto', self.executers)))
            self.log.info("porto is also initialized for container tasks")
        else:
            self.container_executer = None
            self.log.info("porto is not available, multitasks (in containers) are not allowed")

    def _sanitize_task(self, task):
        if isinstance(task, dict):
            task['options']['in_rtc_container'] = self.in_rtc_container
        else:
            # backward compatibility
            task.options['in_rtc_container'] = self.in_rtc_container

    @logged
    def task_started(self, taskid, addr, hostid):
        with self._job_spawned:
            self._awaiting_jobs.discard((taskid, hostid))
            tasks = self.tasks.get(taskid, [])
            task = next(iter(filter(lambda task: task['hostid'] == hostid, reversed(tasks))), None)
            if task is not None:
                task['addr'] = addr
                self._job_spawned.notify_all()

    @logged
    def execute_py2(self, route, local_address, task, extra_arg, path, iface, hostid):
        if self.in_arcadia or self.interpreter_py2 is None or not os.path.exists(self.interpreter_py2):
            uuid = task['uuid']
            exc = CQueueRuntimeError("python2 interpreter is not configured")
            self.log.warning("[%s] task started failed: no python2 interpreter", short(uuid))
            route(result(uuid, 0, dumps((None, exc))), path[:1], hostid=hostid)
            return False

        return self.execute(route,
                            local_address,
                            task,
                            extra_arg,
                            path,
                            hostid,
                            iface,
                            interpreter=self.interpreter_py2)

    @logged
    def execute_py3(self, route, local_address, task, extra_arg, path, iface, hostid):
        if self.in_arcadia or self.interpreter_py3 is None or not os.path.exists(self.interpreter_py3):
            uuid = task['uuid']
            exc = CQueueRuntimeError("python3 interpreter is not configured")
            self.log.warning("[%s] task started failed: no python3 interpreter", short(uuid))
            route(result(uuid, 0, dumps((None, exc))), path[:1], hostid=hostid)
            return False

        return self.execute(route,
                            local_address,
                            task,
                            extra_arg,
                            path,
                            hostid,
                            iface,
                            interpreter=self.interpreter_py3)

    @logged
    def execute(self, route, local_address, task, extra_arg, path, hostid, iface, interpreter=None):
        uuid = task['uuid']
        if not self._precheck_task(uuid, hostid):
            self._send_path_if_needed(route, uuid, path, hostid)
            return False

        if self.verify:
            try:
                self.verify_message(task, task['options']['user'], task['signs'], _task_hash(task), iface)
            except BaseException as e:
                new_exc = self.prepare_exception(e, 'Authentication failure')
                if not isinstance(e, CQueueAuthError):
                    self.log.exception("Error occurred during authentication: {}".format(e), exc_info=sys.exc_info())
                route(result(uuid, 0, dumps((None, new_exc))), path[:1], hostid=hostid)
                return False

        self.log.info('starting task %s from %s (%s@%s) with extra_args (%s)',
                      short(uuid), path[0], task['acc_user'], task['acc_host'], type(extra_arg))

        task['options']['exec_extra_params'] = extra_arg

        with self.user_lock:
            _ensure_tmpdirs(self.has_root)
            taskdir = os.path.join(tempfile.gettempdir(), 'tasks')
            self._sanitize_task(task)
            taskfile = self._write_task(task['options']['user'], task, taskdir, iface)

            return self._try_spawn(
                route,
                local_address,
                task['options']['user'],
                uuid,
                path,
                hostid,
                iface,
                taskfile,
                no_porto=task['options'].get('no_porto', False),
                interpreter=interpreter
            )

    @logged
    def container_execute(self, route, local_address, task, args, path, hostids, iface):
        uuid = task['uuid']
        hostids = list(hostids)
        for idx, hostid in reversed(list(enumerate(hostids))):
            if not self._precheck_task(uuid, hostid):
                hostids.pop(idx)
                self._send_path_if_needed(route, uuid, path, hostid)

        if not hostids:
            return False

        if self.container_executer is None:
            err = CQueueRuntimeError("Porto is not available on this host")
            route(result(uuid, 0, dumps((None, err))), path[:1], hostid=hostids)
            return False

        if self.verify:
            try:
                self.verify_message(task, task['options']['user'], task['signs'], _task_hash(task))
            except BaseException as e:
                new_exc = self.prepare_exception(e, 'Authentication failure')
                if not isinstance(e, CQueueAuthError):
                    self.log.exception("Error occurred during authentication: {}".format(e), exc_info=sys.exc_info())
                route(result(uuid, 0, dumps((None, new_exc))), path[:1], hostid=hostids)
                return False

        taskdir = os.path.join(tempfile.gettempdir(), 'tasks')

        for hostid in hostids:
            container, cont_params = args[hostid]
            if cont_params is None:
                cont_params = {}
            extra_args = cont_params.get('task_params')
            porto_args = cont_params.get('porto_params') or {}
            self.log.info('starting task %s from %s (%s@%s) in container [%s] with extra_args (%s)',
                          short(uuid), path[0], task['acc_user'], task['acc_host'], container, extra_args)

            task['options']['exec_extra_params'] = extra_args

            with self.user_lock:
                _ensure_tmpdirs(self.has_root)
                self._sanitize_task(task)
                taskfile = self._write_task(task['options']['user'], task, taskdir, iface)

                self._try_spawn(
                    route,
                    local_address,
                    task['options']['user'],
                    uuid,
                    path,
                    hostid,
                    iface,
                    taskfile,
                    executer=self.container_executer,
                    container=container,
                    **porto_args
                )

        return True

    @logged
    def portoshell_execute(self, route, local_address, task, args, path, hostids, iface):
        uuid = task['uuid']
        hostids = list(hostids)
        for idx, hostid in reversed(list(enumerate(hostids))):
            if not self._precheck_task(uuid, hostid):
                hostids.pop(idx)
                self._send_path_if_needed(route, uuid, path, hostid)

        if not hostids:
            return False

        # if self.container_executer is None:
        #     self.log.warning("portoshell command requested on host without porto")
        #     err = CQueueRuntimeError("Porto is not available on this host")
        #     route(result(uuid, 0, dumps((None, err))), path[:1], hostid=hostids)
        #     return False

        taskdir = os.path.join(tempfile.gettempdir(), 'tasks')
        task['options']['exec_extra_params'] = ''
        task['options']['exec_fn'] = 'portoshell_slow'

        with self.user_lock:
            _ensure_tmpdirs(self.has_root)

            for hostid in hostids:
                init_msg = args[hostid]
                self.log.info('starting portoshell task %s from %s in slot [%s]',
                              short(uuid),
                              path[0],
                              init_msg['slot'])

                task['data'] = args[hostid]
                self._sanitize_task(task)
                taskfile = self._write_task(None, task, taskdir, iface)

                self._try_spawn(
                    route,
                    local_address,
                    None,
                    uuid,
                    path,
                    hostid,
                    iface,
                    taskfile,
                    interpreter=None,
                    no_porto=True
                )

        return True

    @logged
    def ping(self, route, local_address, task, path, hostid, iface):
        uuid = task['uuid']
        if not self._precheck_task(uuid, hostid):
            self._send_path_if_needed(route, uuid, path, hostid)
            return False

        with self.user_lock:
            _ensure_tmpdirs(self.has_root)
            if isinstance(task, dict):
                task['data'] = ''
                task['options']['exec_extra_params'] = None
                task['options']['exec_fn'] = 'dummy'
            else:
                # backward compatibility
                task.data = ''
                task.options['exec_extra_params'] = None
                task.options['exec_fn'] = 'dummy'

            self.log.info('starting ping %s from %s', short(uuid), path[0])

            taskdir = os.path.join(tempfile.gettempdir(), 'tasks')
            self._sanitize_task(task)
            taskfile = self._write_task(None, task, taskdir, iface)

            return self._try_spawn(route, None, None, uuid, path, hostid, iface, taskfile, interpreter=None)

    @logged
    def report_stats(self, route, task, path, hostid, iface):
        stats = copy.deepcopy(self.stats)
        route(stats_result(stats), path, hostid=hostid)

    def stop_task(self, taskid, hostid):
        self.stopped_tasks.add((taskid, hostid))

        tasks = self.tasks.get(taskid, [])
        with self.user_lock:
            for task in tasks:
                if task['hostid'] == hostid:
                    job = task.get('job', None) if task else None
                    if job:
                        job.terminate()

    def _find_task(self, uuid, hostid):
        tasks = self.tasks.get(uuid)
        if tasks:
            task = next(iter(filter(lambda task: task['hostid'] == hostid, reversed(tasks))), None)
            if task is not None:
                return task

    def _precheck_task(self, uuid, hostid):
        if self._find_task(uuid, hostid):
            self.log.info('drop duplicate task %s[%s]', uuid, hostid)
            return False

        if any(
            forbidden_symbol in uuid
            for forbidden_symbol in './'
        ):
            self.log.warning('possible hijack attempt, task uuid forbidden: %r', uuid)
            return False

        with self._job_ensure_lock:
            # double-checked locking
            if self._find_task(uuid, hostid):
                self.log.info('drop duplicate task %s[%s]', uuid, hostid)
                return False

            self.tasks.setdefault(uuid, []).append({'hostid': hostid})  # just ensure the existance

        if (uuid, hostid) in self.stopped_tasks:
            self.log.info('drop already stopped task %s[%s]', uuid, hostid)
            return False

        return True

    def _send_path_if_needed(self, route, uuid, path, hostid):
        task = self._find_task(uuid, hostid)
        if task:
            addr = task.get('addr')
            if addr:
                route(addpath_msg(uuid, path), [addr], hostid='S')

    def _try_spawn(self,
                   route,
                   local_address,
                   user,
                   uuid,
                   path,
                   hostid,
                   iface,
                   taskfile,
                   executer=None,
                   no_porto=False,
                   interpreter=None,
                   **executer_args):
        interpreter = interpreter or self.interpreter
        executers = [executer] if executer else self.executers
        e = None
        for executer in executers:
            if no_porto and executer.name == 'porto':
                continue

            try:
                self._spawn_job(
                    route,
                    local_address,
                    user,
                    uuid,
                    path,
                    hostid,
                    taskfile,
                    executer=executer,
                    no_porto=no_porto,
                    interpreter=interpreter,
                    iface=iface,
                    **executer_args
                )
            except Exception as ex:
                new_exc = self.prepare_exception(ex, 'Spawn failure')
                e = new_exc  # expose var to function scope
                self.log.warning("failed to spawn task %s with executer %s: %s\n%s",
                                 short(uuid),
                                 executer.name,
                                 e,
                                 ''.join(getTraceback(e)))
            else:
                e = None
                break

        if e is not None:
            route(result(uuid, 0, dumps((None, e))), path, hostid=hostid)
            as_user(user, os.remove, taskfile)
            return False

        return True

    def _spawn_job(self,
                   route,
                   local_address,
                   user,
                   uuid,
                   path,
                   hostid,
                   taskfile,
                   executer,
                   interpreter,
                   iface,
                   no_porto=False,
                   **executer_args):
        args = [interpreter] if self.in_arcadia else [interpreter, self.taskpath]
        args.extend((
            'task',
            '-t', tempfile.gettempdir(),
        ))

        if local_address is not None:
            args.extend(('--notify', local_address))

        extra_env = {'Y_PYTHON_ENTRY_POINT': 'ya.skynet.services.cqudp.main:main'} if self.in_arcadia else None

        args.extend([
            uuid,
            str(hostid),
            base64.encodestring(msgpack.dumps(path)).replace(b'\n', b''),
            taskfile,
        ])

        self.log.info("args: {}".format(args))

        cgroup = self.cgroups_mgr.find_cgroup(user)
        porto_options = self.cgroups_mgr.map_to_porto_limits(cgroup)

        if executer.name == 'procman':
            executer_args['hostid'] = hostid
            if no_porto:
                if not cfg.server.AllowExitContainer:
                    raise RuntimeError("execution not in container is prohibited by host configuration")
                executer_args['porto'] = False
            else:
                executer_args['porto'] = cfg.server.Procman.UsePorto
                executer_args['porto_options'] = porto_options
        elif executer.name == 'porto':
            if no_porto:
                raise ValueError("porto executor isn't valid with no_porto option set")

        tasks = self.tasks.setdefault(uuid, [])
        task = next(iter(filter(lambda t: t['hostid'] == hostid and 'start_time' not in t, reversed(tasks))), None)
        if task is None:
            task = {'start_time': monotime(), 'hostid': hostid}
            tasks.append(task)
        else:
            task['start_time'] = monotime()

        home = getUserHome(user) if user else '/'
        if not os.path.exists(home) or not os.path.isdir(home):
            home = '/'
        job = task['job'] = executer.execute(
            uuid,
            args=args,
            user=user or getUserName(),
            home=home,
            cgroups_path=self.cgroups_mgr.path,
            cgroup=self.cgroups_mgr.find_cgroup(user),
            extra_env=extra_env,
            **executer_args
        )

        self._awaiting_jobs.add((uuid, hostid))
        run_daemon(self._wait_job, route, uuid, hostid, path, job, taskfile, user, iface)

    def _wait_job(self, route, uuid, hostid, path, job, taskfile, user, iface):
        try:
            if job:
                job.join()
        except CQueueRuntimeError as e:
            route(result(uuid, 0, dumps((None, e))), path[:1], hostid=hostid)
        finally:
            exc_info = sys.exc_info()

            self.stats[user]['count'] += 1
            if job:
                self.stats[user]['cpu'] += job.get_cpu()
                self.stats[user]['memory'] = job.get_memory()

            self._remove_task(uuid, user, taskfile, iface)
            if exc_info[0] is not None:
                six.reraise(exc_info[0], exc_info[1], exc_info[2])

    def _write_task(self, user, task, taskdir, iface):
        if user is not None:
            return as_user(user, _write_task, task, taskdir)
        else:
            return _write_task(task, taskdir)

    def _remove_task(self, uuid, user, taskfile, iface):
        try:
            with self.user_lock:
                as_user(user, os.remove, taskfile)
        except EnvironmentError as e:
            if e.errno != errno.ENOENT:
                self.log.exception("[%s] failed to remove task file: %s", short(uuid), e, exc_info=sys.exc_info())

    def shutdown(self):
        super(TaskManager, self).shutdown()
        deadline = monotime() + self.tasks_start_timeout
        with self._job_spawned:
            while self._awaiting_jobs and monotime() < deadline:
                self._job_spawned.wait(min(self.tasks_start_timeout, 2.0))


class InPortoTaskManager(TaskManager):
    in_rtc_container = True

    def __init__(self, *args, **kwargs):
        self.interface_map = kwargs.pop('interface_map')
        kwargs['executers'] = ('porto',)
        super(InPortoTaskManager, self).__init__(*args, **kwargs)
        self.log.debug("initialized with interface map: %s", self.interface_map)

    def select_slot(self, iface):
        return self.interface_map.get(normalize_ip(iface))

    def select_user(self, container, user):
        # For root and other users we check the container isolation and select the right one.
        # But for nobody we skip this, as nobody has no rights and user is already authorized.
        if user in ('nobody',):
            return user

        return self.container_executer.get_container_user(container)[0]

    def verify_message(self, msg, user, signs, hash, iface):
        self.log.debug("verifying task from interface %r", iface)
        return self.mtn_verify_message(msg, user, signs, hash, self.select_slot(iface))

    def _write_task(self, user, task, taskdir, iface):
        slot = self.select_slot(iface)
        container = slot.container
        user = self.select_user(container, user)

        def write_task():
            _ensure_tmpdirs(True)
            return as_user(user, _write_task, task, taskdir)

        with self.user_lock:
            taskfile = self.container_executer.exec_in_container(
                container,
                write_task,
            )
            return taskfile

    def _remove_task(self, uuid, user, taskfile, iface):
        try:
            slot = self.select_slot(iface)
            with self.user_lock:
                self.container_executer.exec_in_container(
                    slot.container,
                    os.remove,
                    taskfile,
                )
        except EnvironmentError as e:
            if e.errno != errno.ENOENT:
                self.log.exception("[%s] failed to remove task file: %s", short(uuid), e, exc_info=sys.exc_info())
        except Exception as e:
            self.log.exception("[%s] failed to remove task file: %s", short(uuid), e, exc_info=sys.exc_info())

    def _try_spawn(
        self,
        route,
        local_address,
        user,
        uuid,
        path,
        hostid,
        iface,
        taskfile,
        executer=None,
        no_porto=False,
        interpreter=None,
        **executer_args
    ):
        e = None
        if executer is not None:
            e = CQueueRuntimeError("Running task with custom executer is prohibited inside the container")
        elif no_porto:
            e = CQueueRuntimeError("Running task not in porto is prohibited inside the container")

        interpreter = interpreter or self.interpreter
        slot = self.select_slot(iface)
        user = self.select_user(slot.container, user)

        if e is not None:
            route(result(uuid, 0, dumps((None, e))), path, hostid=hostid)
            self._remove_task(uuid, user, taskfile, iface)
            return False

        executer_args['container'] = slot.container
        executer_args['home'] = slot.instance_dir

        try:
            self._spawn_job(
                route,
                local_address,
                user,
                uuid,
                path,
                hostid,
                taskfile,
                interpreter=interpreter,
                iface=iface,
                **executer_args
            )
        except Exception as ex:
            new_exc = self.prepare_exception(ex, 'Spawn failure')
            e = new_exc  # expose var to function scope
            self.log.warning("failed to spawn task %s with executer %s: %s\n%s",
                             short(uuid),
                             self.container_executer.name,
                             new_exc,
                             ''.join(getTraceback(new_exc)))

            route(result(uuid, 0, dumps((None, new_exc))), path, hostid=hostid)
            self._remove_task(uuid, user, taskfile, iface)
            return False

        return True

    def _spawn_job(self,
                   route,
                   local_address,
                   user,
                   uuid,
                   path,
                   hostid,
                   taskfile,
                   interpreter,
                   iface,
                   **executer_args):
        args = [interpreter] if self.in_arcadia else [interpreter, self.taskpath]
        args.extend((
            'task',
            '-t', tempfile.gettempdir(),
        ))

        if local_address is not None:
            args.extend(('--notify', local_address))

        args.extend([
            uuid,
            str(hostid),
            base64.encodestring(msgpack.dumps(path)).replace(b'\n', b''),
            taskfile,
        ])

        self.log.info("args: {}".format(args))

        cgroup = self.cgroups_mgr.find_cgroup(user)
        porto_options = self.cgroups_mgr.map_to_porto_limits(cgroup)

        executer_args.update(porto_options)

        tasks = self.tasks.setdefault(uuid, [])
        task = next(iter(filter(lambda t: t['hostid'] == hostid and 'start_time' not in t, reversed(tasks))), None)
        if task is None:
            task = {'start_time': monotime(), 'hostid': hostid}
            tasks.append(task)
        else:
            task['start_time'] = monotime()

        job = task['job'] = self.container_executer.execute(
            uuid,
            args=args,
            user=user,
            **executer_args
        )

        self._awaiting_jobs.add((uuid, hostid))
        run_daemon(self._wait_job, route, uuid, hostid, path, job, taskfile, user, iface)


class _CGroupsManager(object):
    def __init__(self):
        self.allowed = True
        self.cfg = cgroups_cfg.get('tasks', {})

    def find_cgroup(self, user):
        if (
            not self.allowed
            or os.uname()[0].lower() != 'linux'  # no cgroups in freebsd
            or 0 not in os.getresuid()  # no control over cgroups
            or not self.path
            or not self.cfg
        ):
            return None

        target_cgroup = self.cfg.get("ALL")

        if user is None:
            return target_cgroup

        try:
            group = grp.getgrgid(pwd.getpwnam(user).pw_gid).gr_name
        except KeyError:
            pass
        else:
            target_cgroup = self.cfg.get("g@{}".format(group), target_cgroup)

        target_cgroup = self.cfg.get(user, target_cgroup)

        return target_cgroup

    def map_to_porto_limits(self, cgroup):
        if not cgroup or cgroup == '/':
            return {}

        fields = {}

        path = os.path.join(self.path, 'memory', cgroup, 'memory.limit_in_bytes')
        try:
            limit = int(open(path, 'rb').read().strip())
        except Exception:
            pass
        else:
            fields['memory_limit'] = limit

        return fields

    @property
    def path(self):
        return cgroups_cfg.get('path')


def _task_hash(task):
    md5 = hashlib.md5(task['data'])
    md5.update(six.b(task['uuid']))
    md5.update(six.b(task['acc_user']))
    md5.update(six.b(task['acc_host']))
    md5.update(six.b(str(task['ctime'])))

    return md5.digest()


def _write_task(task, taskdir):
    task = dumps(task)
    name = None
    try:
        with tempfile.NamedTemporaryFile(suffix='.task', dir=taskdir, delete=False) as f:
            name = f.name
            f.write(task)
            return name
    except IOError as e:
        if name and os.path.exists(name):
            os.unlink(name)
        if e.errno != errno.ENOSPC:
            raise
        _cleanup(taskdir)

    with tempfile.NamedTemporaryFile(suffix='.task', dir=taskdir, delete=False) as f:
        f.write(task)
        return f.name


def _cleanup(path):
    for filename in os.listdir(path):
        p = os.path.join(path, filename)
        if os.path.isfile(p):
            st = os.stat(p)
            if (time.time() - st.st_mtime) > 600:
                try:
                    as_user(st.st_uid, os.remove, p)
                except EnvironmentError:
                    pass


def _ensure_tmpdirs(has_root=False, root_dir=None):
    def makedir(name, rights=None):
        if not os.path.exists(name):
            os.makedirs(name)
        elif not os.path.isdir(name):
            os.remove(name)
            os.makedirs(name)

        if rights:
            try:
                os.chmod(name, rights)
            except EnvironmentError:  # FreeBSD can deny sticky bit for nonroot
                os.chmod(name, rights & 0o777)

    tempdir = root_dir or tempfile.gettempdir()
    user = 'root' if has_root else None
    as_user(user, makedir, tempdir)
    as_user(user, makedir, os.path.join(tempdir, 'tasks'), 0o1777)
    as_user(user, makedir, os.path.join(tempdir, 'client'), 0o1777)


def stats_result(stats):
    return {
        'uuid': genuuid(),
        'type': 'stats',
        'by_user': stats,
    }


@singleton
def log():
    return root().getChild('taskmgr')
