from __future__ import absolute_import

from ..utils import rotate_list
from itertools import chain

import six
import random


class Envelope(object):
    def __init__(self, msgs, tree, path=None, hostid=None, data=None, tree_data=None, aggr=False):
        self.msgs = [msgs] if not isinstance(msgs, (list, tuple)) else list(msgs)
        self.tree = tree
        self.path = list(path) if path else []
        self.hostid = hostid
        self.data = data
        self.tree_data = tree_data or {}
        self.aggr = aggr

    def can_merge(self, other):
        return (
            self.path == other.path
            and self.msgs == other.msgs
            and self.tree and other.tree
            and not (set(six.iterkeys(self.tree_data)) & set(six.iterkeys(other.tree_data)))
        )

    def merge(self, other):
        self.tree_data.update(other.tree_data)
        self.tree = merge_trees(self.tree, other.tree)

    def initiator(self):
        return self.path[0] if self.path else None

    def delivered(self):
        return not self.tree or None in self.tree

    def next(self):
        if not self.tree:
            return
        for (host, subtree) in self.tree:
            if host is not None:
                yield (
                    host,
                    Envelope(self.msgs,
                             subtree,
                             self.path,
                             hostid=host[0],
                             data=self.tree_data.get(host[0]),
                             tree_data={h: self.tree_data.get(h) for h in subtree_hosts(subtree)},
                             aggr=self.aggr
                             ),
                )

    def __repr__(self):
        return '<Envelope (initiator = {0!s}, msg = {1!r})>'.format(self.initiator(), self.msgs)


def direct_path(h):
    return [(x, None) for x in h]


def subtree_hosts(subtree):
    result = set()
    if not subtree:
        return result
    for host, s in subtree:
        if host is not None:
            result.add(host[0])
            result |= subtree_hosts(s)  # we never make trees deeper than 2
    return result


def replace_master(tree, master):
    new_tree = []
    for h, n in tree:
        if h == master:
            new_tree.append((None, None))
        else:
            new_tree.append((h, n))
    return new_tree


def build_tree(hosts, alive_hosts=None):
    l = len(hosts)

    if l < 25:
        return direct_path(hosts)

    replicas = 2
    steps = int((l / float(replicas)) ** 0.5)

    ret = []

    for step in six.moves.xrange(0, steps):
        part = hosts[(step * l) // steps: ((step + 1) * l) // steps]

        if part:
            for cnt, master in enumerate(random.sample(alive_hosts or part, replicas)):
                shift = (len(part) * cnt) // replicas
                subtree = replace_master(rotate_list(direct_path(part), shift), master)
                ret.append((master, subtree))

    return ret


def build_back_tree(path):
    tree = None
    for h in path[:-1]:
        tree = [(h, tree)]
    return tree


def merge_trees(first, second):
    if not first:
        return second
    elif not second:
        return first

    good_hosts = [host for host, subtree in chain(first, second)]
    all_hosts = list(subtree_hosts(chain(first, second)))
    return build_tree(all_hosts, good_hosts)
