from __future__ import absolute_import
import collections
import threading
import six


class Window(object):
    def __init__(self, maxsize, break_fun=None):
        self.q = collections.deque(maxlen=maxsize)
        self.mutex = threading.Lock()
        self.not_full = threading.Condition(self.mutex)
        self.not_empty = threading.Condition(self.mutex)
        self.break_fun = break_fun if break_fun is not None else (lambda: False)

    def put(self, v):
        self.not_full.acquire()
        try:
            while len(self.q) == self.q.maxlen:
                self.not_full.wait(1.0)
                if self.break_fun():
                    raise WindowTimeout()

            self.q.append(v)
            self.not_empty.notify()

        finally:
            self.not_full.release()

    def front(self):
        with self.not_empty:
            return self.q[0] if len(self.q) else None

    def pop(self):
        with self.not_empty:
            while len(self.q) == 0:
                self.not_empty.wait(1.0)

            v = self.q.popleft()
            self.not_full.notify()
            return v


class IndexedWindow(object):
    def __init__(self, maxsize, break_fun=None):
        self.win = Window(maxsize, break_fun=break_fun)
        self.idx = 0

    def enqueue(self, x):
        self.win.put((self.idx, x))
        self.idx += 1
        return self.idx - 1

    def empty(self):
        return self.win.front() is None

    def seek(self, pos):
        x = self.win.front()
        if not x or pos is None:
            return

        if pos == x[0]:
            return x

        while x and x[0] < pos:
            self.win.pop()
            x = self.win.front()

        if x is not None and pos == x[0]:
            return x


class IncomingWindow(object):
    def __init__(self):
        self.d = collections.deque()
        self.index = 0

    def put(self, idx, value):
        offset = idx - self.index
        if offset < 0:
            return

        if offset >= len(self.d):
            for i in six.moves.xrange(len(self.d), offset + 1):
                self.d.append((i, None))

        self.d[offset] = (idx, value)

    def pop(self):
        return list(self.pull())

    def pull(self):
        while len(self.d) and self.d[0][1] is not None:
            self.index += 1
            yield self.d.popleft()


class MultiWindow(object):
    def __init__(self):
        self.window = IncomingWindow()
        self.indices = collections.defaultdict(int)

    def put(self, index, label, data):
        self.window.put(index, (label, data))

    def pull(self):
        for total_index, (label, data) in self.window.pull():
            local_index = self.indices[label]
            self.indices[label] += 1
            yield total_index, label, local_index, data

    def pop(self):
        return list(self.pull())


class WindowTimeout(Exception):
    pass
