try:
    import os
    import importlib

    import click
    import yaml
    import library.python.vault_client as vault_client
    import yp.data_model as data_model

    from infra.dctl.src import consts
    from infra.dctl.src import version
    from infra.dctl.src.lib import helpers
    from infra.dctl.src.lib.click_quirks import patch_click

    patch_click()

    class LazyProxy(object):
        def __init__(self, modname):
            self.modname = modname

        def __getattr__(self, item):
            obj = importlib.import_module('infra.dctl.src.lib.%s' % self.modname)
            globals()[self.modname] = obj
            return getattr(obj, item)

    cliutil = LazyProxy('cliutil')
    docker_releaser = LazyProxy('docker_releaser')
    endpoint = LazyProxy('endpoint')
    endpoint_set = LazyProxy('endpoint_set')
    stage = LazyProxy('stage')
    project = LazyProxy('project')
    multi_cluster_replica_set = LazyProxy('multi_cluster_replica_set')
    pod = LazyProxy('pod')
    pod_set = LazyProxy('pod_set')
    replica_set = LazyProxy('replica_set')
    dynamic_resource = LazyProxy('dynamic_resource')
    resource_cache = LazyProxy('resource_cache')
    yp_client = LazyProxy('yp_client')
    deploy_ticket = LazyProxy('deploy_ticket')
    release = LazyProxy('release')
    release_rule = LazyProxy('release_rule')
    horizontal_pod_autoscaler = LazyProxy('horizontal_pod_autoscaler')
    subscriptions = LazyProxy('subscriptions')
    stage_draft = LazyProxy('stage_draft')
    approval_policy = LazyProxy('approval_policy')
    notification_policy = LazyProxy('notification_policy')
except KeyboardInterrupt:
    # don't bother user with python stack trace
    # if interrupted (by Ctrl+C) during imports
    raise SystemExit(1)


_color_list = (
    ('True', 'green'),
    ('False', 'red'),
    ('Ready', 'green'),
    ('Closed', 'green'),
    ('InProgress', 'yellow'),
    ('Pending', 'yellow'),
    ('Outdated', 'yellow'),
    ('Unknown', 'red'),
    ('None', 'red'),
    ('STARTED', 'green'),
    ('START_PENDING', 'yellow'),
    ('UNKNOWN', 'red')
)


yaml_loader = getattr(yaml, 'CSafeLoader', yaml.SafeLoader)


class DctlContext(object):
    def __init__(self, dctl_token_path, yp_urls, vault_host):
        self._dctl_token_path = dctl_token_path

        self._yp_urls = yp_urls
        self._yp_token = None
        self._yp_clients = {}

        self._vault_host = vault_host
        self._vault_client = None
        self._vault_client_rsa_fallback = None

    def get_yp_token(self):
        if not self._yp_token:
            self._yp_token = helpers.get_token(consts.YP_TOKEN_ENV,
                                               self._dctl_token_path)
        return self._yp_token

    # TODO: rename to get_yp_client.
    def get_client(self, cluster, address=None, enable_ssl=True):
        """
        :type cluster: str
        :type address: Optional[str]
        :param address: cluster will be ignored if yp_address is provided; meant for using in func tests
        :type enable_ssl: bool
        :param enable_ssl: used to enable/disable ssl for yp connection. meant for using in func tests
        """
        if cluster and address:
            raise click.ClickException('Cannot use cluster and address simultaneously.')
        if address is not None:
            client = yp_client.YpClient(url=address, token=None, enable_ssl=enable_ssl)
            return client

        cluster = cluster.lower()

        if cluster in self._yp_clients:
            return self._yp_clients[cluster]

        url = self._yp_urls[cluster]
        c = yp_client.YpClient(url=url,
                               token=self.get_yp_token())
        self._yp_clients[cluster] = c
        return c

    def get_vault_client(self):
        if self._vault_client:
            return self._vault_client
        # dctl OAuth app has use:vault scope:
        # https://st.yandex-team.ru/DEPLOY-2971
        token = self.get_yp_token()
        self._vault_client = vault_client.VaultClient(
            host=self._vault_host,
            check_status=False,
            authorization=token
        )
        return self._vault_client

    def get_vault_client_rsa_fallback(self):
        if self._vault_client_rsa_fallback:
            return self._vault_client_rsa_fallback
        self._vault_client_rsa_fallback = vault_client.VaultClient(
            host=self._vault_host,
            check_status=False,
            rsa_auth=True,
            rsa_login=cliutil.get_user()
        )
        return self._vault_client_rsa_fallback


aliased_group = cliutil.make_aliased_group_class(consts.COMMAND_ALIASES)
pass_dctl_context = click.make_pass_decorator(DctlContext)


def make_clusters_clients(ctx, clusters):
    """
    :type clusters: list[str]
    :rtype: dict[str, infra.dctl.src.lib.yp_client.YpClient]
    """
    return {c: ctx.get_client(c) for c in clusters}


def add_colors_to_str(string):
    for word, color in _color_list:
        string = str(string).replace(' ' + word + ' ', ' ' + click.style(word, fg=color) + ' ')
    return string


def add_colors_to_table(table):
    """
    :type: prettytable.PrettyTable
    :rtype: str
    """
    table_str_list = str(table).split('\n')

    for i in range(2, len(table_str_list)):  # skip header
        table_str_list[i] = add_colors_to_str(table_str_list[i])

    return '\n'.join(table_str_list)


@click.group(context_settings={'help_option_names': ['-h', '--help']})
@click.option('--token-path', default='~/.dctl/token', help='Path to the dctl OAuth token.')
@click.option('--awacs-url', default='https://awacs.yandex-team.ru', help='IGNORED')
@click.version_option(version.VERSION_MESSAGE, message='%(version)s')
@click.pass_context
def cli(ctx, token_path, awacs_url):
    """
    Welcome to dctl

    To override YP token use DCTL_YP_TOKEN environment variable or
    save path to it in token_path config variable.
    """
    dctl_token_path = os.path.expanduser(token_path)
    ctx.obj = DctlContext(dctl_token_path=dctl_token_path,
                          yp_urls={c.name.lower(): c.address for c in consts.CLUSTER_CONFIGS.values()},
                          vault_host=consts.VAULT_HOST)
    try:
        helpers.check_version(yp_client=ctx.obj.get_client(consts.XDC.name))
    except Exception:
        # Skip exceptions if we cannot check version
        pass


@cli.group(cls=aliased_group, name='upgrade', help='Reserved for ya dctl upgrade')
def upgrade_entry_point():
    raise click.ClickException('"dctl upgrade" can only be used with ya dctl')


@cli.group(cls=aliased_group, name='get', help='Show raw spec of object.')
def get_entry_point():
    pass


@get_entry_point.command(name='pod', help='Show pod content.')
@click.argument('pod_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_pod(ctx, pod_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_POD,
                                object_id=pod_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='pod_set', help='Show podset content.')
@click.argument('pod_set_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_pod_set(ctx, pod_set_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_POD_SET,
                                object_id=pod_set_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='replica_set', help='Show replicaset content.')
@click.argument('replica_set_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_replica_set(ctx, replica_set_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_REPLICA_SET,
                                object_id=replica_set_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='endpoint_set', help='Show endpointset content.')
@click.argument('endpoint_set_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_endpoint_set(ctx, endpoint_set_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_ENDPOINT_SET,
                                object_id=endpoint_set_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='stage', help='Show stage content.')
@click.argument('stage_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS)
)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@click.option('--address', 'yp_address', default=None)
@click.option('--enable-ssl/--disable-ssl', default=True)
@pass_dctl_context
def get_stage(ctx, stage_id, cluster, fmt, skip_ro_fields, yp_address, enable_ssl):
    cluster, yp_address = helpers.validate_cluster_or_address(cluster, yp_address)
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_STAGE,
                                object_id=stage_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt,
                                additional_cleanup=stage.pop_notifications_state,
                                yp_address=yp_address,
                                enable_ssl=enable_ssl
                                )


@get_entry_point.command(name='project', help='Show project content.')
@click.argument('project_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_project(ctx, project_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_PROJECT,
                                object_id=project_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='multi_cluster_replica_set', help='Show multi cluster replicaset content.')
@click.argument('mcrs_id', nargs=1, metavar='multi_cluster_replica_set_id')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_multi_cluster_replica_set(ctx, mcrs_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_MULTI_CLUSTER_REPLICA_SET,
                                object_id=mcrs_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='dynamic_resource', help='Show dynamic resource content.')
@click.argument('dynamic_resource_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_dynamic_resource(ctx, dynamic_resource_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_DYNAMIC_RESOURCE,
                                object_id=dynamic_resource_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='resource_cache', help='Show resource cache content.')
@click.argument('resource_cache_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_resource_cache(ctx, resource_cache_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_RESOURCE_CACHE,
                                object_id=resource_cache_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='horizontal_pod_autoscaler', help='Show horizontal pod autoscaler content.')
@click.argument('horizontal_pod_autoscaler_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_horizontal_pod_autoscaler(ctx, horizontal_pod_autoscaler_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_HORIZONTAL_POD_AUTOSCALER,
                                object_id=horizontal_pod_autoscaler_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='release', help='Show release content.')
@click.argument('release_id', nargs=1)
@click.option('--cluster', '-c',
              type=click.Choice(consts.XDC_CLUSTERS),
              default=consts.XDC_PRODUCTION_CLUSTER)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_release(ctx, release_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_RELEASE,
                                object_id=release_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='release_rule', help='Show release rule content.')
@click.argument('release_rule_id', nargs=1)
@click.option('--cluster', '-c',
              type=click.Choice(consts.XDC_CLUSTERS),
              default=consts.XDC_PRODUCTION_CLUSTER)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_release_rule(ctx, release_rule_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_RELEASE_RULE,
                                object_id=release_rule_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='deploy_ticket', help='Show deploy ticket content.')
@click.argument('ticket_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_deploy_ticket(ctx, ticket_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_DEPLOY_TICKET,
                                object_id=ticket_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='stage_draft', help='Show stage draft content.')
@click.argument('draft_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_stage_draft(ctx, draft_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_STAGE_DRAFT,
                                object_id=draft_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='approval_policy', help='Show approval policy content.')
@click.argument('approval_policy_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_approval_policy(ctx, approval_policy_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_APPROVAL_POLICY,
                                object_id=approval_policy_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@get_entry_point.command(name='notification_policy', help='Show notification policy content.')
@click.argument('notification_policy_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('--format', 'fmt', type=click.Choice(consts.FORMATS), default=consts.DEFAULT_FORMAT)
@click.option('--skip-ro-fields/--keep-ro-fields', default=True)
@pass_dctl_context
def get_notification_policy(ctx, notification_policy_id, cluster, fmt, skip_ro_fields):
    cliutil.get_and_dump_object(ctx=ctx,
                                cluster=cluster,
                                object_type=data_model.OT_NOTIFICATION_POLICY,
                                object_id=notification_policy_id,
                                skip_ro_fields=skip_ro_fields,
                                fmt=fmt)


@cli.group(cls=aliased_group, name='status', help='Show formatted status of object.')
def status_entry_point():
    pass


@status_entry_point.command(name='pod')
@click.argument('pod_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def get_pod_status(ctx, pod_id, cluster):
    client = ctx.get_client(cluster)
    p = pod.get_pod(pod_id, client)
    click.echo(p.status)


@status_entry_point.command(name='replica_set')
@click.argument('replica_set_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@pass_dctl_context
def get_replica_set_status(ctx, replica_set_id, cluster, watch):
    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        t, info = replica_set.get_status(replica_set_id, client)
        click.echo(add_colors_to_table(t))
        click.echo(info)


@status_entry_point.command(name='stage')
@click.argument('stage_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@pass_dctl_context
def get_stage_status(ctx, stage_id, cluster, watch):
    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        result = stage.get_status(stage_id, client)
        click.echo(add_colors_to_table(result))


@status_entry_point.command(name='multi_cluster_replica_set')
@click.argument('mcrs_id', nargs=1, metavar='multi_cluster_replica_set_id')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@pass_dctl_context
def get_multi_cluster_replica_set_status(ctx, mcrs_id, cluster, watch):
    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        t, info = multi_cluster_replica_set.get_status(mcrs_id, client)
        click.echo(add_colors_to_table(t))
        click.echo(info)


@status_entry_point.command(name='dynamic_resource')
@click.argument('dynamic_resource_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@pass_dctl_context
def get_dynamic_resource_status(ctx, dynamic_resource_id, cluster, watch):
    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        spec, status = dynamic_resource.get_status(cluster, dynamic_resource_id, client)
        click.echo(add_colors_to_table(spec))
        click.echo(add_colors_to_table(status))


@status_entry_point.command(name='resource_cache')
@click.argument('resource_cache_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@pass_dctl_context
def get_resource_cache_status(ctx, resource_cache_id, cluster, watch):
    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        spec, statuses = resource_cache.get_status(cluster, resource_cache_id, client)
        click.echo(add_colors_to_table(spec))
        for status in statuses:
            click.echo('')
            click.echo(status[0])
            click.echo(add_colors_to_table(status[1]))


@status_entry_point.command(name='horizontal_pod_autoscaler')
@click.argument('horizontal_pod_autoscaler_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@pass_dctl_context
def get_horizontal_pod_autoscaler_status(ctx, horizontal_pod_autoscaler_id, cluster, watch):
    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        status = horizontal_pod_autoscaler.get_status(cluster, horizontal_pod_autoscaler_id, client)
        click.echo(add_colors_to_table(status))


def _verbose_or_multiple_tickets(ctx, param, value):
    if param == 'verbose' and len(ctx.params.ticket_id) > 1:
        raise click.BadParameter(message="'verbose' cannot be used with multiple ticket ids", ctx=ctx)
    elif param == 'ticket_id' and len(ctx.params.ticket_id) and ctx.params.verbose:
        raise click.BadParameter(message="'verbose' cannot be used with multiple ticket ids", ctx=ctx)

    return value


@status_entry_point.command(name='deploy_ticket', help='Show deploy ticket status.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@click.option(
    '-v', '--verbose',
    is_flag=True,
    help='Also show status for patches in ticket.',
    callback=_verbose_or_multiple_tickets,
)
@click.option(
    '--limit', '-l',
    help='Limit list of tickets to maximum amount',
    default=100,
    type=click.INT,
)
@click.argument(
    'ticket_id',
    nargs=-1,
    callback=_verbose_or_multiple_tickets,
)
@pass_dctl_context
def get_deploy_ticket_status(ctx, ticket_id, cluster, watch, verbose, limit):
    if len(ticket_id) != 1 and verbose:
        raise click.BadOptionUsage(
            message='Verbose mode is supported only with exactly one ticket id specified.',
        )

    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        statuses, patches = deploy_ticket.get_status(
            ticket_ids=ticket_id,
            verbose=verbose,
            limit=limit,
            client=client,
        )

        click.echo(add_colors_to_table(statuses))

        if verbose:
            click.echo('')
            click.echo(add_colors_to_table(patches))


@status_entry_point.command(name='release')
@click.argument('release_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@pass_dctl_context
def get_release_status(ctx, release_id, cluster, watch):
    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        preamble, spec, status = release.get_status(release_id, client)
        click.secho(preamble, fg='yellow')
        click.echo(add_colors_to_table(spec))
        click.echo(add_colors_to_table(status))


@cli.group(cls=aliased_group, name='remove', help='Remove object.')
def remove_entry_point():
    pass


@remove_entry_point.command(name='pod', help='Remove pod.')
@click.argument('pod_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def remove_pod(ctx, pod_id, cluster):
    client = ctx.get_client(cluster)
    pod.remove_pod(pod_id, client)
    click.echo('Removed pod: {}'.format(pod_id))


@remove_entry_point.command(name='pod_set', help='Remove podset.')
@click.argument('pod_set_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def remove_pod_set(ctx, pod_set_id, cluster):
    client = ctx.get_client(cluster)
    pod_set.remove_pod_set(pod_set_id, client)
    click.echo('Removed podset: {}'.format(pod_set_id))


@remove_entry_point.command(name='endpoint_set', help='Remove endpointset.')
@click.argument('endpoint_set_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def remove_endpoint_set(ctx, endpoint_set_id, cluster):
    client = ctx.get_client(cluster)
    endpoint_set.remove(endpoint_set_id, client)
    click.echo('Removed endpointset: {}'.format(endpoint_set_id))


@remove_entry_point.command(name='stage', help='Remove stage.')
@click.argument('stage_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@pass_dctl_context
def remove_stage(ctx, stage_id, cluster):
    client = ctx.get_client(cluster)
    stage.remove(stage_id, client)
    click.echo('Removed stage: {}'.format(stage_id))


@remove_entry_point.command(name='project', help='Remove project.')
@click.argument('project_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@pass_dctl_context
def remove_project(ctx, project_id, cluster):
    client = ctx.get_client(cluster)
    project.remove(project_id, client)
    click.echo('Removed project: {}'.format(project_id))


@remove_entry_point.command(name='replica_set', help='Remove replicaset.')
@click.argument('replica_set_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def remove_replica_set(ctx, replica_set_id, cluster):
    client = ctx.get_client(cluster)
    replica_set.remove_rs(replica_set_id, client)
    click.echo('Removed replicaset: {}'.format(replica_set_id))


@remove_entry_point.command(name='multi_cluster_replica_set', help='Remove multi cluster replicaset.')
@click.argument('mcrs_id', nargs=1, metavar='multi_cluster_replica_set')
@click.option('--cluster', '-c', type=click.Choice(consts.XDC_CLUSTERS), required=True)
@pass_dctl_context
def remove_multi_cluster_replica_set(ctx, mcrs_id, cluster):
    client = ctx.get_client(cluster)
    multi_cluster_replica_set.remove_mcrs(mcrs_id, client)
    click.echo('Removed multi cluster replicaset: {}'.format(mcrs_id))


@remove_entry_point.command(name='dynamic_resource', help='Remove dynamic resource.')
@click.argument('dynamic_resource_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def remove_dynamic_resource(ctx, dynamic_resource_id, cluster):
    client = ctx.get_client(cluster)
    dynamic_resource.remove(dynamic_resource_id, client)
    click.echo('Removed dynamic resource: {}'.format(dynamic_resource_id))


@remove_entry_point.command(name='resource_cache', help='Remove resource cache.')
@click.argument('resource_cache_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def remove_resource_cache(ctx, resource_cache_id, cluster):
    client = ctx.get_client(cluster)
    resource_cache.remove(resource_cache_id, client)
    click.echo('Removed resource cache: {}'.format(resource_cache_id))


@remove_entry_point.command(name='horizontal_pod_autoscaler', help='Remove horizontal pod autoscaler.')
@click.argument('horizontal_pod_autoscaler_id', nargs=1)
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def remove_horizontal_pod_autoscaler(ctx, horizontal_pod_autoscaler_id, cluster):
    client = ctx.get_client(cluster)
    horizontal_pod_autoscaler.remove(horizontal_pod_autoscaler_id, client)
    click.echo('Removed horizontal pod autoscaler: {}'.format(horizontal_pod_autoscaler_id))


@remove_entry_point.command(name='release_rule', help='Remove release rule.')
@click.argument('release_rule_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
)
@pass_dctl_context
def remove_release_rule(ctx, release_rule_id, cluster):
    client = ctx.get_client(cluster)
    release_rule.remove(release_rule_id, client)
    click.echo('Removed release rule: {}'.format(release_rule_id))


@remove_entry_point.command(name='subscription', help='Remove events subscription.')
@click.option(
    '--user', '-u',
    help='Remove subscription of given user',
)
@click.option(
    '--stage-id',
    help='Remove subscription to the given stage',
)
def remove_subscription(user, stage_id):
    if not user:
        user = cliutil.get_user()

    subscriptions.unsubscribe(user=user, stage_id=stage_id)
    click.secho('Unsubscribed.', fg='green')


@remove_entry_point.command(name='approval_policy', help='Remove approval policy.')
@click.argument('approval_policy_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
)
@pass_dctl_context
def remove_approval_policy(ctx, approval_policy_id, cluster):
    client = ctx.get_client(cluster)
    approval_policy.remove(approval_policy_id, client)
    click.echo('Removed approval policy: {}'.format(approval_policy_id))


@remove_entry_point.command(name='notification_policy', help='Remove notification policy.')
@click.argument('notification_policy_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
)
@pass_dctl_context
def remove_notification_policy(ctx, notification_policy_id, cluster):
    client = ctx.get_client(cluster)
    notification_policy.remove(notification_policy_id, client)
    click.echo('Removed notification policy: {}'.format(notification_policy_id))


@cli.group(cls=aliased_group, name='list', help='Show list of objects.')
def list_entry_point():
    pass


@list_entry_point.command(name='pod', help='Show list of pods related to replicaset.')
@click.argument('replica_set_id', nargs=1)
@click.option(
    '--cluster', '-c',
    multiple=True,
    type=click.Choice(consts.CLUSTERS),
    default=consts.PRODUCTION_CLUSTERS,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_pods(ctx, replica_set_id, cluster,  watch, limit):
    clients = make_clusters_clients(ctx, cluster)
    for _ in cliutil.watching(watch):
        result = pod.list_objects(replica_set_id, clients, limit)
        click.echo(add_colors_to_table(result))


@list_entry_point.command(name='replica_set', help='Show list of replicasets.')
@click.option(
    '--cluster', '-c',
    multiple=True,
    type=click.Choice(consts.CLUSTERS),
    default=consts.PRODUCTION_CLUSTERS,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_replica_sets(ctx, cluster, user, all_entities, limit):
    clients = make_clusters_clients(ctx, cluster)
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    result = replica_set.list_objects(clients, user, limit)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='stage', help='Show list of stages.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
    help='default={}'.format(consts.XDC_PRODUCTION_CLUSTER)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@click.option('--project', '-p', help='Print entities related to the specified project.')
@pass_dctl_context
def list_stages(ctx, cluster, user, all_entities, limit, project):
    client = ctx.get_client(cluster)
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    result = stage.list_objects(client, cluster, user, limit, project)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='project', help='Show list of projects.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
    help='default={}'.format(consts.XDC_PRODUCTION_CLUSTER)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_projects(ctx, cluster, user, all_entities, limit):
    client = ctx.get_client(cluster)
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    result = project.list_objects(client, cluster, user, limit)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='multi_cluster_replica_set', help='Show list of multi cluster replicasets.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
    help='default={}'.format(consts.XDC_PRODUCTION_CLUSTER)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@click.option('-w', '--watch', is_flag=True, help='Auto update the screen every 5 seconds.')
@pass_dctl_context
def list_multi_cluster_replica_sets(ctx, cluster, user, all_entities, watch, limit):
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    client = ctx.get_client(cluster)
    for _ in cliutil.watching(watch):
        result = multi_cluster_replica_set.list_objects(client, cluster, user, limit)
        click.echo(add_colors_to_table(result))


@list_entry_point.command(name='pod_set', help='Show list of pod_sets.')
@click.option(
    '--cluster', '-c',
    multiple=True,
    type=click.Choice(consts.CLUSTERS),
    default=consts.PRODUCTION_CLUSTERS,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_pod_sets(ctx, cluster, user, all_entities, limit):
    clients = make_clusters_clients(ctx, cluster)
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    result = pod_set.list_objects(clients, user, limit)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='endpoint_set', help='Show list of endpoint_sets.')
@click.option(
    '--cluster', '-c',
    multiple=True,
    type=click.Choice(consts.CLUSTERS),
    default=consts.PRODUCTION_CLUSTERS,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_endpoint_sets(ctx, cluster, user, all_entities, limit):
    clients = make_clusters_clients(ctx, cluster)
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    result = endpoint_set.list_objects(clients, user, limit)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='endpoint', help='Show list of endpoints related to endpoint_set.')
@click.argument('endpoint_set_id', nargs=1)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@click.option(
    '--cluster', '-c',
    multiple=True,
    type=click.Choice(consts.CLUSTERS),
    default=consts.PRODUCTION_CLUSTERS,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@pass_dctl_context
def list_endpoints(ctx, endpoint_set_id, cluster, limit):
    clients = make_clusters_clients(ctx, cluster)
    result = endpoint.list_objects(clients, endpoint_set_id, limit)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='dynamic_resource', help='Show list of dynamic resources.')
@click.option(
    '--cluster', '-c',
    multiple=True,
    type=click.Choice(consts.CLUSTERS),
    default=consts.PRODUCTION_CLUSTERS,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@click.option('--podset', '--pod-set', '-p', 'pod_set', help='Print entities related to the specified pod_set.')
@pass_dctl_context
def list_dynamic_resources(ctx, cluster, user, all_entities, pod_set, limit):
    clients = make_clusters_clients(ctx, cluster)
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    result = dynamic_resource.list_objects(clients, user, pod_set, limit)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='resource_cache', help='Show list of resource caches.')
@click.option(
    '--cluster', '-c',
    multiple=True,
    type=click.Choice(consts.CLUSTERS),
    default=consts.PRODUCTION_CLUSTERS,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@click.option('--podset', '--pod-set', '-p', 'pod_set', help='Print entities related to the specified pod_set.')
@pass_dctl_context
def list_resource_caches(ctx, cluster, user, all_entities, pod_set, limit):
    clients = make_clusters_clients(ctx, cluster)
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    result = resource_cache.list_objects(clients, user, pod_set, limit)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='horizontal_pod_autoscaler', help='Show list of horizontal pod autoscalers.')
@click.option(
    '--cluster', '-c',
    multiple=True,
    type=click.Choice(consts.CLUSTERS),
    default=consts.PRODUCTION_CLUSTERS,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@click.option('--user', '-u', help='Print entities owned by given user (default is current user).')
@click.option('--all', '-a', 'all_entities', help='Print all the entities, not only owned by given user.', is_flag=True)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@click.option('--replicaset', '--replica-set', '-p', 'replica_set', help='Print entities related to the specified replica_set.')
@pass_dctl_context
def list_horizontal_pod_autoscalers(ctx, cluster, user, all_entities, replica_set, limit):
    clients = make_clusters_clients(ctx, cluster)
    if all_entities and user:
        raise click.ClickException('Cannot use --all/-a option and --user/-u simultaneously.')
    if not all_entities and not user:
        user = cliutil.get_user()
    result = horizontal_pod_autoscaler.list_objects(clients, user, replica_set, limit)
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='release_rule', help='Show list of release rules.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
    help='default={}'.format(consts.XDC_PRODUCTION_CLUSTER),
)
@click.option(
    '--user', '-u',
    help='Print entities owned by given user (default is current user).',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['user', 'all'],
    optional_group=True,
)
@click.option(
    '--stage', '-s',
    help='Print entities belonging to the specified stage.',
)
@click.option(
    '--recipe-id', '-r',
    help='Print entities belonging to the specified recipe.',
)
@click.option(
    '--all', '-a',
    help='Print all the entities, not only owned by given user.',
    is_flag=True,
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['user', 'all'],
    optional_group=True,
)
@click.option(
    '--sandbox-task-type',
    help='Print entities belonging to the given sandbox task.',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['sandbox_task_type', 'docker_image'],
    optional_group=True,
)
@click.option(
    '--docker-image',
    help='Print entities belonging to the given docker image.',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['sandbox_task_type', 'docker_image'],
    optional_group=True,
)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_release_rules(
    ctx,
    cluster,
    user,
    all,
    stage,
    recipe_id,
    sandbox_task_type,
    docker_image,
    limit,
):
    client = ctx.get_client(cluster)
    if not all and not user:
        user = cliutil.get_user()

    result = release_rule.list_objects(
        client=client,
        user=user,
        stage=stage,
        limit=limit,
        recipe_id=recipe_id,
        sandbox_task_type=sandbox_task_type,
        docker_image=docker_image,
    )
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='approval_policy', help='Show list of approval policies.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
    help='default={}'.format(consts.XDC_PRODUCTION_CLUSTER),
)
@click.option(
    '--user', '-u',
    help='Print entities owned by given user (default is current user).',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['user', 'all'],
    optional_group=True,
)
@click.option(
    '--stage', '-s',
    help='Print entities belonging to the specified stage.',
)
@click.option(
    '--all', '-a',
    help='Print all the entities, not only owned by given user.',
    is_flag=True,
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['user', 'all'],
    optional_group=True,
)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_approval_policies(
    ctx,
    cluster,
    user,
    all,
    stage,
    limit,
):
    client = ctx.get_client(cluster)
    if not all and not user:
        user = cliutil.get_user()

    result = approval_policy.list_objects(
        client=client,
        user=user,
        stage=stage,
        limit=limit
    )
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='notification_policy', help='Show list of notification policies.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
    help='default={}'.format(consts.XDC_PRODUCTION_CLUSTER),
)
@click.option(
    '--user', '-u',
    help='Print entities owned by given user (default is current user).',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['user', 'all'],
    optional_group=True,
)
@click.option(
    '--stage', '-s',
    help='Print entities belonging to the specified stage.',
)
@click.option(
    '--all', '-a',
    help='Print all the entities, not only owned by given user.',
    is_flag=True,
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['user', 'all'],
    optional_group=True,
)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_notification_policies(
    ctx,
    cluster,
    user,
    all,
    stage,
    limit,
):
    client = ctx.get_client(cluster)
    if not all and not user:
        user = cliutil.get_user()

    result = notification_policy.list_objects(
        client=client,
        user=user,
        stage=stage,
        limit=limit
    )
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='release', help='Show list of releases.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
    help='default={}'.format(consts.XDC_PRODUCTION_CLUSTER),
)
@click.option(
    '--release-type',
    help='Print only releases with the given release type.',
)
@click.option(
    '--release-author',
    help='Print only releases with the given author.',
)
@click.option(
    '--sandbox-task-type',
    help='Print entities belonging to the given sandbox task.',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['sandbox_task_type', 'docker_image'],
    optional_group=True,
)
@click.option(
    '--docker-image',
    help='Print entities belonging to the given docker image.',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['sandbox_task_type', 'docker_image'],
    optional_group=True,
)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_releases(
    ctx,
    cluster,
    release_type,
    release_author,
    sandbox_task_type,
    docker_image,
    limit,
):
    client = ctx.get_client(cluster)

    result = release.list_objects(
        client=client,
        release_type=release_type,
        release_author=release_author,
        limit=limit,
        sandbox_task_type=sandbox_task_type,
        docker_image=docker_image,
    )
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='deploy_ticket', help='Show list of deploy tickets.')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER,
    help='default={}'.format(consts.XDC_PRODUCTION_CLUSTER),
)
@click.option(
    '--stage', '-s',
    help='Print entities belonging to the specified stage.',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['stage', 'release_rule'],
    optional_group=True,
)
@click.option(
    '--release-rule', '-r',
    help='Print entities belonging to the specified release rule.',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['stage', 'release_rule'],
    optional_group=True,
)
@click.option(
    '--pending',
    help='Print only pending tickets.',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['pending', 'active'],
    optional_group=True,
)
@click.option(
    '--active',
    help='Print only active tickets.',
    cls=cliutil.MutuallyExclusiveOption,
    exclusive_names=['pending', 'active'],
    optional_group=True,
)
@click.option('--limit', '-l', type=click.INT, default=100, help='Entities count limit (default 100).')
@pass_dctl_context
def list_deploy_tickets(
    ctx,
    cluster,
    stage,
    release_rule,
    pending,
    active,
    limit,
):
    client = ctx.get_client(cluster)

    result = deploy_ticket.list_objects(
        client=client,
        stage=stage,
        limit=limit,
        release_rule=release_rule,
        pending=pending,
        active=active,
    )
    click.echo(add_colors_to_table(result))


@list_entry_point.command(name='subscription', help='Show list of stage event subscriptions.')
@click.option(
    '--user', '-u',
    help='Print subscriptions of given user',
)
def list_subscriptions(user):
    if not user:
        user = cliutil.get_user()

    result = subscriptions.list_subscriptions(user=user)
    click.echo(add_colors_to_table(result))


@cli.group(cls=aliased_group, name='put', help='Put object.')
def put_entry_point():
    pass


@put_entry_point.command(name='replica_set')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.CLUSTERS),
    multiple=True,
    required=True,
    help='cluster to create replica set (can be used multiple times)'
)
@pass_dctl_context
def put_replica_set(ctx, yml, cluster):
    d = yaml.load(yml, Loader=yaml_loader)
    rs = replica_set.cast_yaml_dict_to_yp_object(d)
    clients = make_clusters_clients(ctx, cluster)
    for c, client in clients.items():
        replica_set.put_rs(rs, client)
        click.secho('[{}] Put replica set: {}'.format(c, rs.meta.id), fg='green')


@put_entry_point.command(name='multi_cluster_replica_set')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    required=True
)
@pass_dctl_context
def put_multi_cluster_replica_set(ctx, yml, cluster):
    d = yaml.load(yml, Loader=yaml_loader)
    mcrs = multi_cluster_replica_set.cast_yaml_dict_to_yp_object(d)
    client = ctx.get_client(cluster)
    multi_cluster_replica_set.put_mcrs(mcrs, client)
    msg = 'Put multi cluster replica set: {}'.format(mcrs.meta.id)
    click.secho(msg, fg='green')


@put_entry_point.command(name='stage')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS)
)
@click.option('--rewrite-delegation-tokens', default=False, is_flag=True)
@click.option('--address', 'yp_address', default=None)
@click.option('--enable-ssl/--disable-ssl', default=True)
@pass_dctl_context
def put_stage(ctx, yml, cluster, rewrite_delegation_tokens, yp_address, enable_ssl):
    cluster, yp_address = helpers.validate_cluster_or_address(cluster, yp_address)
    d = yaml.load(yml, Loader=yaml_loader)
    st = stage.cast_yaml_dict_to_yp_object(d)
    stage.validate(st)
    stage.put(stage=st,
              cluster=cluster,
              rewrite_delegation_tokens=rewrite_delegation_tokens,
              vault_client=ctx.get_vault_client(),
              vault_client_rsa_fallback=ctx.get_vault_client_rsa_fallback(),
              client=ctx.get_client(cluster, yp_address, enable_ssl))
    click.secho('Put stage: {}'.format(st.meta.id), fg='green')


@put_entry_point.command(name='project')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@pass_dctl_context
def put_project(ctx, yml, cluster):
    d = yaml.load(yml, Loader=yaml_loader)
    pr = project.cast_yaml_dict_to_yp_object(d)
    project.validate(pr)
    client = ctx.get_client(cluster)
    project.put(pr, client)
    click.secho('Put project: {}'.format(pr.meta.id), fg='green')


@put_entry_point.command(name='pod')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.CLUSTERS),
    required=True,
)
@pass_dctl_context
def put_pod(ctx, yml, cluster):
    pod_as_dict = yaml.load(yml, Loader=yaml.SafeLoader)
    pod_dto = pod.cast_yaml_dict_to_yp_object(pod_as_dict)
    client = ctx.get_client(cluster)
    pod.put(pod_dto, client)
    click.secho('Put pod: {}'.format(pod_dto.meta.id), fg='green')


@put_entry_point.command(name='endpoint_set')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.CLUSTERS),
    multiple=True,
    required=True,
    help='default={}, (can be used multiple times)'.format(consts.PRODUCTION_CLUSTERS)
)
@pass_dctl_context
def put_endpoint_set(ctx, yml, cluster):
    d = yaml.load(yml, Loader=yaml_loader)
    es = endpoint_set.cast_yaml_dict_to_yp_object(d)
    clients = make_clusters_clients(ctx, cluster)
    for c, client in clients.items():
        endpoint_set.put(es, client)
        click.secho('[{}] Put endpoint set: {}'.format(c, es.meta.id), fg='green')


@put_entry_point.command(name='dynamic_resource')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def put_dynamic_resource(ctx, yml, cluster):
    client = ctx.get_client(cluster)
    d = yaml.load(yml, Loader=yaml_loader)
    dr = dynamic_resource.cast_yaml_dict_to_yp_object(d)
    dynamic_resource.put(dr, client)
    click.secho('Put dynamic_resource: {}'.format(dr.meta.id), fg='green')


@put_entry_point.command(name='resource_cache')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def put_resource_cache(ctx, yml, cluster):
    client = ctx.get_client(cluster)
    d = yaml.load(yml, Loader=yaml_loader)
    dr = resource_cache.cast_yaml_dict_to_yp_object(d)
    resource_cache.put(dr, client)
    click.secho('Put resource_cache: {}'.format(dr.meta.id), fg='green')


@put_entry_point.command(name='horizontal_pod_autoscaler')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option('--cluster', '-c', type=click.Choice(consts.CLUSTERS), required=True)
@pass_dctl_context
def put_horizontal_pod_autoscaler(ctx, yml, cluster):
    client = ctx.get_client(cluster)
    d = yaml.load(yml, Loader=yaml_loader)
    dr = horizontal_pod_autoscaler.cast_yaml_dict_to_yp_object(d)
    horizontal_pod_autoscaler.put(dr, client)
    click.secho('Put horizontal_pod_autoscaler: {}'.format(dr.meta.id), fg='green')


@put_entry_point.command(name='release_rule')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@pass_dctl_context
def put_release_rule(ctx, yml, cluster):
    client = ctx.get_client(cluster)
    d = yaml.load(yml, Loader=yaml_loader)
    rr = release_rule.cast_yaml_dict_to_yp_object(d)
    release_rule.put(rr, client)
    click.secho('Put release rule: {}'.format(rr.meta.id), fg='green')


@put_entry_point.command(name='subscription', help='Add events subscription.')
@click.option(
    '--user', '-u',
    help='Add subscription for given user',
)
@click.option(
    '--stage-id',
    help='Add subscription to the given stage',
)
@click.option(
    '--telegram/--no-telegram',
    help='Enable notifications to telegram',
    is_flag=True,
    default=None,
)
@click.option(
    '--email/--no-email',
    help='Enable notifications by email',
    is_flag=True,
    default=None,
)
def put_subscription(
    user,
    stage_id,
    telegram,
    email,
):
    if not user:
        user = cliutil.get_user()

    subscriptions.subscribe(
        user=user,
        stage_id=stage_id,
        enable_telegram=telegram,
        enable_email=email,
    )
    click.secho('Subscribed.', fg='green')


@put_entry_point.command(name='approval_policy')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@pass_dctl_context
def put_approval_policy(ctx, yml, cluster):
    client = ctx.get_client(cluster)
    d = yaml.load(yml, Loader=yaml_loader)
    ap = approval_policy.cast_yaml_dict_to_yp_object(d)
    approval_policy.put(ap, client)
    click.secho('Put approval policy: {}'.format(ap.meta.id), fg='green')


@put_entry_point.command(name='notification_policy')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@pass_dctl_context
def put_notification_policy(ctx, yml, cluster):
    client = ctx.get_client(cluster)
    d = yaml.load(yml, Loader=yaml_loader)
    ap = notification_policy.cast_yaml_dict_to_yp_object(d)
    notification_policy.put(ap, client)
    click.secho('Put notification policy: {}'.format(ap.meta.id), fg='green')


@cli.group(cls=aliased_group, name='copy', help='Copy object.')
def copy_entry_point():
    pass


@copy_entry_point.command(name='stage')
@click.argument('yml', nargs=1, type=click.File('rb'))
@click.argument('new_stage_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@pass_dctl_context
def copy_stage(ctx, yml, new_stage_id, cluster):
    d = yaml.load(yml, Loader=yaml_loader)
    st = stage.cast_yaml_dict_to_yp_object(d)
    client = ctx.get_client(cluster)
    stage.copy_stage(stage=st,
                     new_stage_id=new_stage_id,
                     vault_client=ctx.get_vault_client(),
                     vault_client_rsa_fallback=ctx.get_vault_client_rsa_fallback(),
                     cluster=cluster,
                     client=client)
    click.secho('Copied stage: {}'.format(st.meta.id), fg='green')


@cli.group(cls=aliased_group, name='commit', help='Commit action.')
def commit_entry_point():
    pass


@commit_entry_point.command(name='deploy_ticket')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('--reason', '-r', default='DCTL_COMMIT', help='Machine-readable commit message')
@click.option('--message', '-m', default='Commit from dctl', help='Human-readable commit message')
@click.option('--patch-id', multiple=True, help='One or more patches for partial commit')
@click.argument('ticket_id', nargs=1)
@pass_dctl_context
def commit_deploy_ticket(ctx, cluster, reason, message, patch_id, ticket_id):
    client = ctx.get_client(cluster)
    deploy_ticket.commit(
        client=client,
        reason=reason,
        message=message,
        ticket_id=ticket_id,
        patches=patch_id
    )

    click.secho('Commited ticket: {}'.format(ticket_id), fg='green')


@cli.group(cls=aliased_group, name='skip', help='Skip action.')
def skip_entry_point():
    pass


@skip_entry_point.command(name='deploy_ticket')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option('--reason', '-r', default='DCTL_COMMIT', help='Machine-readable skip message')
@click.option('--message', '-m', default='Commit from dctl', help='Human-readable skip message')
@click.option('--patch-id', multiple=True, help='One or more patches for partial skip')
@click.argument('ticket_id', nargs=1)
@pass_dctl_context
def skip_deploy_ticket(ctx, cluster, reason, message, patch_id, ticket_id):
    client = ctx.get_client(cluster)
    deploy_ticket.skip(
        client=client,
        reason=reason,
        message=message,
        ticket_id=ticket_id,
        patches=patch_id
    )

    click.secho('Skipped ticket: {}'.format(ticket_id), fg='green')


@cli.group(cls=aliased_group, name='release', help='Create release from specified entity')
def release_entry_point():
    pass


@release_entry_point.command(name='docker')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option(
    '--registry',
    default='registry.yandex.net',
)
@click.option(
    '--tag', '-t',
    help='Tag to make release from.',
    required=True,
)
@click.option(
    '--release-type',
    help='Release type (testing, stable, etc.)',
    default='testing',
)
@click.option(
    '--title',
    help='Release title.',
)
@click.option(
    '--description',
    help='Release description.',
)
@click.argument('image_name', nargs=1)
@pass_dctl_context
def release_docker(ctx, cluster, registry, image_name, tag, release_type, title, description):
    client = ctx.get_client(cluster)
    try:
        release = docker_releaser.release(client, registry, image_name, tag, release_type, title, description)
    except Exception as e:
        click.secho("Failed to release: {}".format(e), fg='red')
        raise click.Abort()

    click.secho(
        "Created release {!r} for {}/{}:{} (hash {})".format(
            release.meta.id,
            release.spec.docker.registry,
            release.spec.docker.image_name,
            release.spec.docker.image_tag,
            release.spec.docker.image_hash,
        ),
        fg='green'
    )


@cli.group(cls=aliased_group, name='publish-draft', help='Publish stage draft.')
def publish_entry_point():
    pass


@cli.group(cls=aliased_group, name='sidecar', help='Stage infra component sidecars.')
def sidecar_entry_point():
    pass


@sidecar_entry_point.command(name='update', help='Update sidecars.')
@click.argument('stage_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS)
)
@click.option('--address', 'yp_address', default=None)
@click.option('--enable-ssl/--disable-ssl', default=True)
@pass_dctl_context
def sidecar_update(
    ctx,
    stage_id,
    cluster,
    yp_address,
    enable_ssl
):
    cluster, yp_address = helpers.validate_cluster_or_address(cluster, yp_address)
    client = ctx.get_client(cluster, yp_address, enable_ssl)
    stage.update_sidecars(client, stage_id)


@publish_entry_point.command(name='stage')
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.argument(
    'yml',
    nargs=1,
    type=click.File('rb'),
)
@click.option(
    '--message', '-m',
    default='Publish from dctl',
    help='Human-readable publish message'
)
@pass_dctl_context
def publish_draft(ctx, cluster, yml, message):
    client = ctx.get_client(cluster)
    stage_dict = yaml.load(yml, Loader=yaml_loader)
    s = stage.cast_yaml_dict_to_yp_object(stage_dict)
    draft_id, ticket_id = stage_draft.create_draft_with_deploy_ticket(
        client=client,
        stage=s,
        title=message
    )

    yd_url = consts.YD_URLS_BY_XDC_CLUSTERS[cluster]
    click.secho(
        'Created and published new stage draft: '
        '{}/stage/{}/releases/{}'.format(
            yd_url,
            s.meta.id,
            ticket_id
        ),
        fg='green'
    )


@cli.group(cls=aliased_group, name='control', help='Control object.')
def control_entry_point():
    pass


@control_entry_point.group(cls=aliased_group, name='stage', help='Control object.')
def control_stage_entry_point():
    pass


@control_stage_entry_point.command(name='override_deployment_strategy', help='Add events overriding deployment strategy.')
@click.argument('stage_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option(
    '--revision',
    help='Revision that will be overrided.',
    required=True,
    type=int
)
@click.option(
    '--deploy-unit-id',
    help='Deploy unit that will be overrided.',
    required=True
)
@click.option(
    '--max-unavailable',
    help='New max unavailable value for deployment strategy.',
    required=True,
    type=int
)
@click.option(
    '--cluster-to-override',
    help='Clusters to override.',
    multiple=True,
    required=True
)
@pass_dctl_context
def override_deployment_strategy(
        ctx,
        stage_id,
        cluster,
        revision,
        deploy_unit_id,
        max_unavailable,
        cluster_to_override
):
    client = ctx.get_client(cluster)
    stage.override_deployment_strategy(client, stage_id, revision, deploy_unit_id, max_unavailable, cluster_to_override)


@control_stage_entry_point.command(name='approve', help='Approves the location for deploy.')
@click.argument('stage_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option(
    '--deploy-revision',
    help='Deploy unit revision.',
    required=True,
    type=int
)
@click.option(
    '--deploy-unit-id',
    help='Deploy unit id.',
    required=True
)
@click.option(
    '--clusters',
    help='Clusters to approve.',
    multiple=True,
    required=True
)
@pass_dctl_context
def approve_cluster(
    ctx,
    stage_id,
    cluster,
    deploy_revision,
    deploy_unit_id,
    clusters
):
    client = ctx.get_client(cluster)
    stage.approve_location(client, stage_id, deploy_revision, deploy_unit_id, clusters)


@control_stage_entry_point.command(name='disapprove', help='Disapproves the location for deploy.')
@click.argument('stage_id', nargs=1)
@click.option(
    '--cluster', '-c',
    type=click.Choice(consts.XDC_CLUSTERS),
    default=consts.XDC_PRODUCTION_CLUSTER
)
@click.option(
    '--deploy-revision',
    help='Deploy unit revision.',
    required=True,
    type=int
)
@click.option(
    '--deploy-unit-id',
    help='Deploy unit id.',
    required=True
)
@click.option(
    '--clusters',
    help='Clusters to disapprove.',
    multiple=True,
    required=True
)
@pass_dctl_context
def disapprove_cluster(
    ctx,
    stage_id,
    cluster,
    deploy_revision,
    deploy_unit_id,
    clusters
):
    client = ctx.get_client(cluster)
    stage.disapprove_location(client, stage_id, deploy_revision, deploy_unit_id, clusters)
