import getpass
import json
import time

import click
import yaml

import yp.common
import yt_yson_bindings
import yt.yson as yson
import yp.data_model as data_model
from infra.dctl.src import consts


yaml_dumper = getattr(yaml, 'CSafeDumper', yaml.SafeDumper)


def get_user():
    return getpass.getuser()


def watching(watch):
    if watch:
        while True:
            click.clear()
            yield
            time.sleep(consts.WATCH_TIME_INTERVAL)
    else:
        yield


def make_aliased_group_class(aliases):
    reverse_dict = {}
    for command, names in aliases.items():
        reverse_dict[command] = command
        for a in names:
            if a in reverse_dict:
                raise ValueError('Same aliases for different commands')
            reverse_dict[a] = command

    class AliasedGroup(click.Group):

        def get_command(self, ctx, cmd_name):
            real_cmd = reverse_dict.get(cmd_name, cmd_name)
            return click.Group.get_command(self, ctx, real_cmd)

        def command(self, *args, **kwargs):
            names = aliases.get(kwargs['name'])
            if names:
                kwargs['help'] = '%s %s' % (names, kwargs.get('help', ''))
            return super(AliasedGroup, self).command(*args, **kwargs)

    return AliasedGroup


def clear_annotated_fields(m, annotation_name):
    """
    :type m: yp.data_model.TMeta
    :type annotation_name: str
    :rtype: None
    """
    for field in m.DESCRIPTOR.fields:
        for f, value in field.GetOptions().ListFields():
            if f.name == annotation_name and value:
                m.ClearField(field.name)


def clear_not_updatable_fields(obj):
    obj.ClearField('status')
    clear_annotated_fields(obj.meta, 'not_updatable')


def clear_not_initializable_fields(obj):
    obj.ClearField('status')
    clear_annotated_fields(obj.meta, 'not_initializable')


def dump_yp_object_to_yaml(o):
    """
    :type o: yp.data_model.T
    :rtype: str
    """
    yson_s = yson.loads(yt_yson_bindings.dumps_proto(o))

    # NOTE: a little hack here: use json dumps/loads for casting all attributes
    # to strings.
    yson_s = json.loads(json.dumps(yson_s))
    return yaml.dump(yson_s, Dumper=yaml_dumper)


def dump_yp_object_dict_to_yaml(d):
    """
    :type d: dict
    """
    # NOTE: use yson_to_json to avoid "YsonEntity is not JSON serializable" error
    d = yson.yson_to_json(d)
    return yaml.dump(d, Dumper=yaml_dumper)


def stringify_status_from_conditions(status):
    def stringify_condition(condition, fg, default_message):
        if condition.status == data_model.CS_TRUE:
            return click.style(
                condition.message or condition.reason or default_message,
                fg=fg,
            )

    return (
        stringify_condition(status.failed, 'red', 'Failed')
        or stringify_condition(status.in_progress, 'blue', 'InProgress')
        or stringify_condition(status.ready, 'green', 'Ready')
        or "Unknown"
    )


def stringify_condition_as_status(condition):
    if condition.status == data_model.CS_TRUE:
        return click.style('Ready', fg='green')
    return click.style('InProgress', fg='blue')


def stringify_condition_as_succeeded(condition):
    if condition.status == data_model.CS_TRUE:
        return click.style('OK', fg='green')

    err = condition.reason or condition.message or 'Failed'
    return click.style(err, fg='red')


def get_and_dump_object(ctx, cluster, object_type, object_id, skip_ro_fields, fmt, additional_cleanup=None,
                        yp_address=None, enable_ssl=True):
    """
    :type cluster: unicode
    :type object_type: int
    :type object_id: unicode
    :type skip_ro_fields: bool
    :type fmt: unicode
    :type additional_cleanup: Optional[Callable]
    :type yp_address: Optional[str]
    :param yp_address: cluster will be ignored if yp_address is provided; meant for using in func tests
    :type enable_ssl: bool
    :param enable_ssl: used to enable/disable ssl for yp connection. meant for using in func tests
    """
    client = ctx.get_client(cluster, yp_address, enable_ssl=enable_ssl)
    p = client.get(object_type=object_type, object_id=object_id)
    if skip_ro_fields:
        clear_not_initializable_fields(p)
    p = yp.common.protobuf_to_dict(p)
    if additional_cleanup is not None:
        additional_cleanup(p)
    if object_type == data_model.OT_PROJECT or object_type == data_model.OT_STAGE:
        p.get('spec').pop('account_id', None)
    if fmt == 'yaml':
        p = dump_yp_object_dict_to_yaml(p)
    click.echo(p)


class MutuallyExclusive(object):
    def __init__(self, *args, **kwargs):
        self.optional_group = kwargs.pop('optional_group', False)

        exclusive_names = set(kwargs.pop('exclusive_names') or ())
        super(MutuallyExclusive, self).__init__(*args, **kwargs)

        exclusive_names.discard(self.name)
        if not len(exclusive_names):
            raise click.BadParameter(
                message="MutuallyExclusiveOption must have at least one conflicting option name",
            )
        self.exclusive_names = exclusive_names


class MutuallyExclusiveArgument(MutuallyExclusive, click.Argument):
    def handle_parse_result(self, ctx, opts, args):
        conflicting = self.exclusive_names & set(opts)
        value_present = bool(opts.get(self.name))
        if conflicting and value_present:
            raise click.BadOptionUsage(
                message="%r argument is mutually exclusive with: %s" % (self.name, ", ".join(conflicting)),
                ctx=ctx,
            )
        elif not conflicting and not value_present and not self.optional_group:
            raise click.UsageError(
                message="at least one option from the set must be specified: %s" % (
                    ", ".join(self.exclusive_names | {self.name})
                ),
                ctx=ctx,
            )

        return super(MutuallyExclusiveArgument, self).handle_parse_result(ctx, opts, args)


class MutuallyExclusiveOption(MutuallyExclusive, click.Option):
    def handle_parse_result(self, ctx, opts, args):
        conflicting = self.exclusive_names & set(opts)
        conflicting = {
            name
            for name in (self.exclusive_names & set(opts))
            if opts.get(name)
        }
        if conflicting and self.name in opts:
            raise click.BadOptionUsage(
                message="%r option is mutually exclusive with: %s" % (self.name, ", ".join(conflicting)),
                ctx=ctx,
            )
        elif not conflicting and self.name not in opts and not self.optional_group:
            raise click.UsageError(
                message="at least one option from the set must be specified: %s" % (
                    ", ".join(self.exclusive_names | {self.name})
                ),
                ctx=ctx,
            )

        return super(MutuallyExclusiveOption, self).handle_parse_result(ctx, opts, args)
