import click
import prettytable
from yp import data_model


def get(ticket_id, client):
    return client.get(
        object_type=data_model.OT_DEPLOY_TICKET,
        object_id=ticket_id,
    )


def list_tickets(client, limit):
    return client.list(
        object_type=data_model.OT_DEPLOY_TICKET,
        user=None,
        limit=limit,
    )


def commit(client, reason, message, ticket_id, patches):
    return client.commit_ticket(
        object_id=ticket_id,
        reason=reason,
        message=message,
        patches=patches,
    )


def skip(client, reason, message, ticket_id, patches):
    return client.skip_ticket(
        object_id=ticket_id,
        reason=reason,
        message=message,
        patches=patches,
    )


def get_ticket_progress_str(progress):
    ticket_status = click.style('Pending', fg='blue')
    if progress.in_progress.status == data_model.CS_TRUE:
        ticket_status = click.style('InProgress', fg='yellow')
    elif progress.closed.status == data_model.CS_TRUE:
        ticket_status = click.style('Closed', fg='green')

    return ticket_status


def get_patch_progress_str(progress):
    ticket_status = click.style('Pending', fg='blue')
    if progress.in_progress.status == data_model.CS_TRUE:
        ticket_status = click.style('InProgress', fg='brown')
    elif progress.success.status == data_model.CS_TRUE:
        ticket_status = click.style('Success', fg='green')
    elif progress.failed.status == data_model.CS_TRUE:
        ticket_status = click.style('Failed', fg='red')
    elif progress.cancelled.status == data_model.CS_TRUE:
        ticket_status = click.style('Cancelled', fg='yellow')

    return ticket_status


def get_action_str(action):
    return {
        data_model.DPAT_NONE: "NONE",
        data_model.DPAT_COMMIT: "COMMIT",
        data_model.DPAT_SKIP: "SKIP",
        data_model.DPAT_ON_HOLD: "ON HOLD",
        data_model.DPAT_WAIT: "WAIT",
    }.get(action, "UNKNOWN")


def get_status(ticket_ids, verbose, limit, client):
    assert not (len(ticket_ids) > 1 and verbose)

    status = prettytable.PrettyTable([
        'TicketId',
        'Action',
        'Status',
    ])

    patches = None

    if ticket_ids:
        tickets = [get(ticket_id, client) for ticket_id in ticket_ids]
    else:
        tickets = list_tickets(client, limit)

    for ticket in tickets:
        status.add_row([
            ticket.meta.id,
            get_action_str(ticket.status.action.type),
            get_ticket_progress_str(ticket.status.progress),
        ])

        if verbose and not patches:
            patches = prettytable.PrettyTable([
                'PatchId',
                'Action',
                'Status',
            ])

            for patch_id, patch_info in ticket.status.patches.items():
                patches.add_row([
                    patch_id,
                    get_action_str(patch_info.action.type),
                    get_patch_progress_str(patch_info.progress),
                ])

    return status, patches


def list_objects(
    client,
    stage,
    release_rule,
    pending,
    active,
    limit,
):
    query_pieces = []
    if stage:
        query_pieces.append('[/meta/stage_id] = "%s"' % (stage,))
    if release_rule:
        query_pieces.append('[/spec/release_rule_id] = "%s"' % (release_rule,))
    if pending:
        query_pieces.append('[/status/progress/pending/status] = "true"')
    if active:
        query_pieces.append('[/status/progress/in_progress/status] = "true"')

    query = ' AND '.join(query_pieces) or None

    table = prettytable.PrettyTable(['ID', 'Stage', 'Release', 'ReleaseRule', 'Status'])

    for dt in client.list(
        object_type=data_model.OT_DEPLOY_TICKET,
        user=None,
        limit=limit,
        query=query,
    ):
        table.add_row([
            dt.meta.id,
            dt.meta.stage_id,
            dt.spec.release_id,
            dt.spec.release_rule_id,
            get_ticket_progress_str(dt.status.progress),
        ])

    return table
